/*
Battery Capacity Meter Firmware
Author: Mauro Grassi, February 2009 for Silicon Chip Publications.
Notes: by Mauro Grassi.

Important Note: Compilation Instructions for the C18 compiler:
- All optimizations ON (Procedural Abstraction 2 passes) with banking optimizer OFF & extended instruction set!
- Multi Bank Large Data Model Small Code Model as well as large stack (-ls option to compiler)
(M.G.)

USB may lead to crashes of host computer if the bypass capacitor on the Vusb pin is not big enough in value. 
I have been using a 1uF and it works well. With a 100nF instead you will get constant
crashing of host PC when USB plugged in. 
Note that the datasheet recommends 220nF certainly don't use any smaller than this.

Versions up to 4.00 were on a breadboard with slightly different pin outs, especially for LCD and AN channels.
Versions from 4.00 are for the PC board.

Version 4.30: 	first version with full coherent RLE and USB functioning correctly (fixed a small bug in the goOutOfStandBy routine
				was calling USBConnect() regardless of whether the USB was already connected.

Version 5.60: 	this is the first version with extended instruction set enabled, to save considerable program space.

Version 5.70: 	this version has averaging (from 0- 120 samples can be averaged on all ADC channels!) When 0 is selected, we use
			  	a running average for all channels.
			  
Version 6.10: 	this is a good version, adds logging times, RLE buffer position & log mode.

Version 6.15: 	this fixes the 0xFFFF pop problem for logTotalTime.

Version 7.00: 	Important Note: Versions before 7.00 ie up to 6.99 where for the prototype that did not have independent control of the Relay output.
			  	Versions after and including 7.00 have independent control of the Relay output, RELAY is controlled by RA4 and POWERON=RA5 now!
			  	The voltage at the USB pin cannot be monitored in versions from 7.00 onwards, but is not used anyway. The firmware adds support for
			  	a shutdown voltage as before, but also for a load switch off voltage (usually higher than the shutdown voltage). The relay turns on
			  	again if the voltage rises above the load switch off voltage as well, with hysteresis added.
*/

#include <p18cxxx.h>
#include "typedefs.h"                   
#include "io_cfg.h"
#include "usb.h"
#include "batterymeter.h"
#include <stdlib.h>
#include <math.h>

#if   defined(__18F4550)||defined(__18F4455)|| \
	  defined(__18F2550)||defined(__18F2455)|| \
      defined(__18F4553)||defined(__18F4458)|| \
      defined(__18F2553)||defined(__18F2458)

#pragma config PLLDIV   = 5       		// (20 MHz input)
#pragma config CPUDIV   = OSC1_PLL2		//
#pragma config USBDIV   = 2       		// Clock source from 96MHz PLL/2
#pragma config FOSC     = HSPLL_HS		//
#pragma config FCMEN    = OFF
#pragma config IESO     = OFF
#pragma config PWRT     = ON
#pragma config BOR      = ON
#pragma config BORV     = 3
#pragma config VREGEN   = ON
#pragma config WDT      = OFF
#pragma config WDTPS    = 32768
#pragma config MCLRE    = OFF			// not using MCLR as reset pin = OFF
#pragma config LPT1OSC  = OFF			// you could enable this if you want.
#pragma config PBADEN   = OFF
#pragma config CCP2MX   = ON
#pragma config STVREN   = ON			// Stack Overflow Reset Enabled!!
#pragma config LVP      = OFF
//#pragma config ICPRT    = OFF       	// Dedicated In-Circuit Debug/Programming
#pragma config XINST    = ON       		// Extended Instruction Set
#pragma config CP0      = OFF
#pragma config CP1      = OFF
#pragma config CP2      = OFF
#pragma config CP3      = OFF
#pragma config CPB      = OFF
#pragma config CPD      = OFF
#pragma config WRT0     = OFF
#pragma config WRT1     = OFF
#pragma config WRT2     = OFF
#pragma config WRT3     = OFF
#pragma config WRTB     = OFF      		// Boot Block Write Protection
#pragma config WRTC     = OFF
#pragma config WRTD     = OFF
#pragma config EBTR0    = OFF
#pragma config EBTR1    = OFF
#pragma config EBTR2    = OFF
#pragma config EBTR3    = OFF
#pragma config EBTRB    = OFF
//
#endif
// Declarations
extern byte ReadEEPROM(byte);
extern byte WriteEEPROM(byte, byte);
byte 		translateKeyCode(byte);
//byte 		translateNumberCode(byte);
void ReadEEPROMF(byte, byte*);
void WriteEEPROMF(byte, byte*);
byte 		areYouSure(void);
void 		clearWithoutUpdate(void);
void 		updateLogChannelStrings(byte);
void 		displayDisplayMode(byte);
void 		USBConnect(void);
void 		saveSettings(void);
void 		restoreSettings(void);
void 		shiftGeneral(int, int, int, int, byte*);
void 		enterNumberString(byte, ram char*, int);
void 		doBeep(int);
void 		idF(int);
void 		restoreAllDefaults(void);
void 		declareFullCapacity(void);
void 		resetTimeOut(void);
void 		setFullCapacity(int);
void 		setVoltageRail50(int);
//void		calibrateVoltageRail50(int);
//void 		setVoltageRail33(int);
void        retrieveRLEData(byte*, byte*);
void        detectSenseResistance(int);
void 		setThreshold(int);
void        setCheckPeriod(int);
void 		setTrickle(int);
void 		setPeukerts(int);
void 		setLogError(int);
void		setDetectionPoints(int);
void        setLogFloor(int);
void 		setLogPeriod(int);
void 		setChemistry(int);
void 		setMinVoltage(int);
void 		setMaxVoltage(int);
void		setAvgCount(int);
void 		setDividerLow(int);
void 		setCutOffVoltage(int);
void 		setAlarm(int);
void 		setDividerHigh(int);
void 		setChargingEfficiency(int);
void 		seeFirmwareVersion(int);
void        seeRx(int);
//void        seeRLEBuffer(int);
void        seeRLEPosition(int);
void 		seeUSBStatus(int);
void 		seeBatteryPercent(int);
void		seeCycles(int);
void		setCyclePercent(int);
void 		seeBatteryCapacityAH(int);
void 		seeBatteryVoltage(int);
void 		seeLoadCurrent(int);
void 		seeChargeCurrent(int);
void 		seeCircuitCurrent(int);
void		seeChargeAndVoltage(int);
void 		seeNetBatteryDrain(int);
void 		seeBatteryTimeMinutes(int);
void 		setBeeperStatus(int);
//
void		setRelayHysteresis(int);
void		setRelayOffVoltage(int);
void		setCurrentMaxTimeOutPeriod(int);
void		setLoadCurrentMax(int);
void		setRelayEnable(int);
//
void 		setTimeOutSeconds(int);
void 		setBrightnessPercent(int);
void 		setLoadDifferentialGain(int);
void        setChargeDifferentialGain(int);
void 		seeVoltageRail50(int);
//void 		seeVUSBRail(int);
void		updateRelayState(void);
void 		seeCircuitVoltage(int);
void 		seeLoadDiffAmpVoltage(int);
void 		seeChargeDiffAmpVoltage(int);
void		seeLastSync(int);
void        seeLogTime(int);
void        seeLogTotalTime(int);
void 		setShuntResistance(int);
void		calibrateCurrentSystem(int);
void		setRelayCoilResistance(int);
void        setLogMode(int);
void		seeFuseString(int);
void		turnPowerOff(int);
void 		setCh1(int);
void 		setCh2(int);
void 		setCh3(int);
void 		setCh4(int);
void 		setSenseResistance(int);
void 		setBrightnessLevel(double);
void        writeRLEString(byte);
void 		writeCellTypeString(byte);
void		writeFuseString(byte);
rom MENU    detectionMenu;
rom MENU	relayMenu;
rom MENU 	batterySettingsMenu;
rom MENU 	systemSettingsMenu;
rom MENU 	displayMenu;
rom MENU 	calibrationSettingsMenu;
rom MENU 	mainMenu;
rom MENU 	systemStatusMenu;
rom MENU* 	executeMenu(byte, rom MENU*);
rom MENU 	adcSystemStatusMenu;
void 		displayDisplayMode(byte);
// Global Variables
#pragma udata gpr0=0x60
// System Variables
volatile 	byte 		noKeys;
volatile 	byte 		lcdScreen[LCD_SIZE];
volatile 	byte 		oldlcdScreen[LCD_SIZE];
char 		buffer[NUM_DIGITS+4];
byte 		keyPtr;
byte 		keyFull;
// this is a 4 byte boundary
volatile	double 		time;
volatile	double 		logPeriodCounter;
double      checkPeriod;                // in seconds to check for sync.
double 		logPeriod;
double 		logError;                   // relative error
double 		alarmCapacity;
double 		cutOffVoltage;				// voltage below which the circuit cuts off power completely
//volatile 	double 		vusbRail;
//double 			    voltageRail33;				// +3.3V Rail (Volts)
double 					voltageRail50;				// +5V 	 Rail (Volts)
double 					dividerLow;					// voltage divider for the lower of the two battery voltages
double 					dividerHigh;				// voltage divider for the higher of the two battery voltages
double 					peukertsConstant;			// a constant
double 		            currentThreshold;			// in Amps to go into standby mode
byte 		displayMode;
byte 		keyBuffer[KEY_BUFFER_SIZE];
byte 		keyGetPtr;
volatile 	byte 		brightness;
volatile 	byte 		setBrightness;
volatile 	SAMPLE 		sample[NUM_SAMPLES-2];
// Relay Control Variables
volatile	double		relayHysteresis;			// voltage hysteresis for relay on off automatic control
//
volatile	byte 		totalUpdated;
volatile	unsigned int samplingCount;
volatile 	byte 		samplingIndex;
volatile	byte 		alarmOn;
char   					cellTypeString[10];			// cell chemistry string either "Nickel" or "Lead Acid"
volatile 	char   		yesNoUSBString[8];			// this string is either 'Yes' or 'No'
volatile	char   		yesNoBeeperString[4];		// this string is either 'Yes' or 'No'
byte   		beepOn;						// 1= beeper enabled 0= beeper disabled
byte		cellType;
rom MENU* 	menuStack[MENU_STACK_SIZE];
byte 		menuStartIndexStack[MENU_STACK_SIZE];
byte 		menuPtr;
byte 		menuFull;
// Data Logging Subsystem Data & Buffer
volatile 	double 		RLEData[RLE_CHANNELS];
volatile 	unsigned int 	RLEMultiplicity[RLE_CHANNELS];
volatile	byte 			RLEBuffer[HALF_RLE_BUFFER_SIZE];
byte 		logChannel[RLE_CHANNELS];
char 		logChannelString[RLE_CHANNELS][8];
volatile	double		itime;
volatile 	int 			RLEGetPtr;
volatile	double		loadCurrentMax;				// overcurrent protection for relay off...
double		detectionPoints;
volatile	double 		chargeDiffAmpVoltage;
volatile	double 		loadDiffAmpVoltage;
volatile	double 		circuitVoltage;				// in Volts
volatile	double 		batteryPercent;				// battery percent charge
volatile	double 		batteryCapacityAH;			// battery capacity in AH (Amp Hours)
volatile	double 		batteryVoltage;				// in Volts
volatile	double 		loadCurrent;				// in Amps
volatile	double 		circuitCurrent;				// in mAmps
volatile	double 		netCurrent;					// net current drain in Amps
volatile	double 		batteryMinutes;				// remaining Capacity
volatile	byte		netCurrentSign;
volatile	double 		chargeCurrent;				// charging current in A
double			 		loadDifferentialGain;		// gain of the front end differential current sense chip for load
double                  chargeDifferentialGain;     // gain of the front end differential current sense chip for charge
volatile	double 		shuntResistance;			// in milli Ohms
double 					senseResistance;			// in the circuit to measure the circuit current drain (in Ohms)
volatile	double 		timeOutSecondsCounter;
double 					timeOutInSeconds;
double			 		brightnessPercent;
double 					chargingEfficiency;			// a percentage
double 					fullCapacity;				// in AH (Amp Hours)
volatile	double 		capacityAs;
volatile	double		currentMaxTimeOut;
volatile	double		currentMaxTimeOutPeriod;
volatile	byte		relayOn;					// bit 0= on/off control
volatile	byte		currentFuse;				
volatile	byte		relayEnable;				// bit 0= global disable or enable relay...
char		relayString[4];
char		fuseString[6];
int 		lcdTransitionType;
volatile	double		relayOffVoltage;			// voltage below which relay turns off
double      logFloor;                   // absolute error

byte 		dummy;

#pragma udata tempbffr=0x4A0
byte 		tempbuffer[TEMPBUFF_SIZE];

#pragma udata usb5=0x500
volatile 	byte 		scanLn;
volatile 	byte 		scanCode;
volatile 	byte 		scanRaw;
volatile 	byte 		scanCounter;
volatile	int 		RLEPtr;
volatile	int 		RLEFull;
double 					minVoltage;					// voltage at which the battery can be considered fully discharged
double 					maxVoltage;					// voltage at which the battery can be considered fully charged (when trickle charging)
volatile	double		avgSampleCount;
volatile 	double  	lastSync;
volatile	double		wattHours;
volatile 	double 		loadWatts;
volatile 	double 		lastChargeCurrent;
volatile	double 		lastBatteryVoltage;
double 		trickleCurrent;
double 		rX;							// this is the resistance of the wire connecting the BATT+ shunt terminal to the
										// PC board BATT+ terminal!
unsigned	int 		timerDelay;
volatile 	byte 		RLEBuffer2[HALF_RLE_BUFFER_SIZE];
int 		lastCall;
volatile    byte 		lastUSB;
byte 		lastmode;
volatile	char   		timeString[20];				// this is the time string in dXXhXXm format (days, hours, minutes)
volatile    char        RLEString[5];
volatile    double      RLETime;                    // time elapsed since the buffer is full...
volatile    double      RLETotalTime;
byte        RLEMode;
double 		            firmwareVersion;			// firmware version as double X.XX (2 decimal places are significant)
double		cycles;									// number of charge discharge cycles
double		cyclePercent;							// threshold for cycle detection
byte 		stby;
byte		icycle;									// there are three bands icycle=0 when batteryPercent is in the top cyclePercent, 1= in the middle, =2 when in the bottom cyclePercent
double		relayCoilResistance;					// resistance of the relay coil...
double		relayCurrent;							// in mA
volatile	byte 		standBy;
volatile	byte 		enableStandByDetect;

#pragma code

rom MENU calibrationSettingsMenu=
{
	9,
	{
		{ 0, { (rom char*)"Set +5V Rail (V)", 				setVoltageRail50,		&voltageRail50, 	2, 		0 } },
		{ 0, { (rom char*)"Calibrate Lo. Divider", 	 		setDividerLow,   		&dividerLow,		2, 		0 } },
		{ 0, { (rom char*)"Calibrate Hi. Divider", 	 		setDividerHigh,	    	&dividerHigh, 		2, 		0 } },
		{ 0, { (rom char*)"Shunt Resistance (m\xF4)", 		setShuntResistance,   	&shuntResistance, 	3,		0 } },
		{ 0, { (rom char*)"Calibrate Current", 		 		calibrateCurrentSystem, 0, 	                0,	    0 } },
		{ 0, { (rom char*)"Load Amp. Gain", 				setLoadDifferentialGain, &loadDifferentialGain, 	2,		0 } },
		{ 0, { (rom char*)"Charge Amp. Gain", 				setChargeDifferentialGain, &chargeDifferentialGain, 	2,		0 } },
		{ 0, { (rom char*)"Sense Resistance (\xF4)", 		setSenseResistance,     &senseResistance, 	1,		0 } },
		{ 0, { (rom char*)"Detect Sense Resistance",        detectSenseResistance,  0,                  0,      0 } }
	}
};

rom MENU batterySettingsMenu=
{
	6,
	{
		{ 0, { "Declare Full Now", 			declareFullCapacity ,	0, 						0, 		0 } },
		{ 0, { "Battery Capacity (AH)",		setFullCapacity , 		&fullCapacity, 			1, 		0 } },
		{ 0, { "Peukert's Constant", 		setPeukerts,			&peukertsConstant, 		3, 		0 } },
		{ 0, { "Cell Chemistry", 			setChemistry, 			0, 						0, 		&cellTypeString[0] } },
		{ 0, { "Charging Efficiency (%)", 	setChargingEfficiency,	&chargingEfficiency,	1, 		0 } },
		{ (rom MENU*)0, { (rom char*)"Cycle Threshold (%)", setCyclePercent,		&cyclePercent,		0, 0 } },
	}
};

rom MENU detectionMenu=
{
	9,
	{ 
		{ (rom MENU*)0, { (rom char*)"Shutdown Voltage (V)", 	setCutOffVoltage, &cutOffVoltage, 1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Capacity Alarm (%)", 		setAlarm, &alarmCapacity, 1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Standby Threshold (A)", 	setThreshold, &currentThreshold, 3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Trickle Current (A)", 	setTrickle, &trickleCurrent, 3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Min. Voltage (V)", 		setMinVoltage, 			&minVoltage, 			1, 		0 } },
		{ (rom MENU*)0, { (rom char*)"Max. Voltage (V)", 		setMaxVoltage, 			&maxVoltage, 			1, 		0 } },
		{ (rom MENU*)0, { (rom char*)"Detection Period (sec)", 	setCheckPeriod, &checkPeriod, -1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Detection Points",		setDetectionPoints, &detectionPoints, 0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Last Sync.",				seeLastSync, 0, 0, 0 } }
	}
};

rom MENU relayMenu=
{
	7,
	{ 
		{ (rom MENU*)0, { (rom char*)"Relay System", 			setRelayEnable, 0, 0, &relayString[0] } },
		{ (rom MENU*)0, { (rom char*)"Relay Coil Resistance (\xF4)", 	setRelayCoilResistance, &relayCoilResistance, 1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Relay Off Voltage (V)", 	setRelayOffVoltage, &relayOffVoltage, 2, 0 } },
		{ (rom MENU*)0, { (rom char*)"Relay Hysteresis (V)", 	setRelayHysteresis, &relayHysteresis, 2, 0 } },
		{ (rom MENU*)0, { (rom char*)"Max. Load Current (A)", 	setLoadCurrentMax,  &loadCurrentMax,  3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Max. Current Duration (sec)", 	setCurrentMaxTimeOutPeriod, &currentMaxTimeOutPeriod, 1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Overload Fuse Status", 	seeFuseString, 0, 0, &fuseString[0] } }
	}
};

rom MENU systemSettingsMenu=
{
	9,
	{ 
		{ (rom MENU*)0, { (rom char*)"Beeper Status", 			setBeeperStatus, 0, 0, &yesNoBeeperString[0] } },
		{ (rom MENU*)0, { (rom char*)"Average Samples", 		setAvgCount, (double*)&avgSampleCount, -1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Firmware Version", 		seeFirmwareVersion, &firmwareVersion, -2, 0 } },	// unsigned 2 decimal places
		{ (rom MENU*)0, { (rom char*)"USB Connected", 			seeUSBStatus, 0, 0, &yesNoUSBString[0] } },
		{ (rom MENU*)0, { (rom char*)"Last Sync.",				seeLastSync, 0, 0, 0 } }, 
		{ (rom MENU*)0, { (rom char*)"Last Log Sample",			seeLogTime,  0, 0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Log Total Time",			seeLogTotalTime,  0, 0, 0 } },
//	    { (rom MENU*)0, { (rom char*)"Buffer Position",         seeRLEPosition,     0,              0, 0 } },
		{ (MENU*)0,	  						{ (rom char*)"Restore Defaults",    restoreAllDefaults, 0, 0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Turn Meter Off",			turnPowerOff, 0, 0, 0 } }

	}
};

rom MENU systemStatusMenu=
{
	9,
	{ 
		{ (rom MENU*)0, { (rom char*)"Batt. Charge (%)", 	seeBatteryPercent,    	&batteryPercent,		1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Batt. Capacity (AH)", seeBatteryCapacityAH, 	&batteryCapacityAH, 	1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Batt. Voltage (V)", 	seeBatteryVoltage, 	  	&batteryVoltage, 		1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Load (A)", 			seeLoadCurrent, 	  	&loadCurrent,			3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Charge (A)", 			seeChargeCurrent, 	  	&chargeCurrent,			3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Circuit (mA)", 		seeCircuitCurrent, 	  	&circuitCurrent,		0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Net (A)", 			seeNetBatteryDrain,   	&netCurrent,			3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Time (mins)", 		seeBatteryTimeMinutes,	&batteryMinutes,		0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Cycles",				seeCycles,				&cycles,				0, 0 } },
	}
};

rom MENU adcSystemStatusMenu=
{
	6,
	{ 
		{ (rom MENU*)0, { (rom char*)"+5.0V Rail (V)", 				seeVoltageRail50,    	&voltageRail50,			2, 0 } },
//		{ (rom MENU*)0, { (rom char*)"+3.3V Rail (V)",	 			seeVUSBRail,       		&vusbRail,     			2, 0 } },
		{ (rom MENU*)0, { (rom char*)"Batt.Voltage (V)", 			seeBatteryVoltage, 	  	&batteryVoltage, 		1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Circuit Voltage (V)", 		seeCircuitVoltage,		&circuitVoltage,		1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Load Diff. Amp Voltage (V)", 	seeLoadDiffAmpVoltage,	&loadDiffAmpVoltage,	3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Chrg.Diff. Amp Voltage (V)", 	seeChargeDiffAmpVoltage,&chargeDiffAmpVoltage,	3, 0 } },
        { (rom MENU*)0, { (rom char*)"Internal Resistance",         seeRx,                  &rX,                    6, 0 } }
     
	}
};

rom MENU logSettingsMenu=
{
	9,
	{ 
		{ (rom MENU*)0, { (rom char*)"RLE Error (%)", 			setLogError,		&logError,   	2, 0 } },
		{ (rom MENU*)0, { (rom char*)"RLE Absolute Error", 		setLogFloor,		&logFloor,   	3, 0 } },
		{ (rom MENU*)0, { (rom char*)"Sampling Period (sec)", 	setLogPeriod, 		&logPeriod,	 	1, 0 } },
		{ (rom MENU*)0, { (rom char*)"Ch.1 Logs", 		        setCh1, 			0, 	0, &logChannelString[0][0] } },
		{ (rom MENU*)0, { (rom char*)"Ch.2 Logs", 		        setCh2, 			0,	0, &logChannelString[1][0] } },
		{ (rom MENU*)0, { (rom char*)"Ch.3 Logs", 		        setCh3, 			0,	0, &logChannelString[2][0] } },
		{ (rom MENU*)0, { (rom char*)"Ch.4 Logs", 		        setCh4, 			0,	0, &logChannelString[3][0] } },
        { (rom MENU*)0, { (rom char*)"Log Mode",                setLogMode,         0,  0, &RLEString[0] } },
	    { (rom MENU*)0, { (rom char*)"RLE Buffer Position",     seeRLEPosition,     0,              0, 0 } },
	}
};

rom MENU displayMenu=
{
	2,
	{ 
		{ (rom MENU*)0, { (rom char*)"Brightness (%)",  setBrightnessPercent,	&brightnessPercent, 0, 0 } },
		{ (rom MENU*)0, { (rom char*)"Timeout (sec)",   setTimeOutSeconds, 		&timeOutInSeconds, 	0, 0 } },
	}
};

rom MENU mainMenu=
{
	9,
	{ 
		{ (MENU*)&batterySettingsMenu, 		{ (rom char*)"Battery Menu",        0, 0, 0, 0 } },
		{ (MENU*)&calibrationSettingsMenu, 	{ (rom char*)"Calibration Menu",    0, 0, 0, 0 } },
		{ (MENU*)&displayMenu, 				{ (rom char*)"Display Menu",        0, 0, 0, 0 } },
		{ (MENU*)&detectionMenu,            { (rom char*)"Detection Menu",      0, 0, 0, 0 } },
		{ (MENU*)&systemSettingsMenu,  		{ (rom char*)"System Menu",         0, 0, 0, 0 } },
		{ (MENU*)&relayMenu, 				{ (rom char*)"Relay Menu", 0, 0, 0, 0 } },
		{ (MENU*)&logSettingsMenu,  		{ (rom char*)"Logging Menu",        0, 0, 0, 0 } },
		{ (MENU*)&systemStatusMenu,         { (rom char*)"Battery Status",      0, 0, 0, 0 } },
		{ (MENU*)&adcSystemStatusMenu, 		{ (rom char*)"ADC Status", 0, 0, 0, 0 } },
	}
};

void myISRHigh(void);
void myISR(void);

#pragma code

#pragma code _HIGH_INTERRUPT_VECTOR = 0x000008
void _high_ISR (void)
{
    _asm goto myISRHigh _endasm
	//_asm goto RM_HIGH_INTERRUPT_VECTOR _endasm
}

#pragma code _LOW_INTERRUPT_VECTOR = 0x000018
void _low_ISR (void)
{
    _asm goto myISR _endasm
	//_asm goto RM_LOW_INTERRUPT_VECTOR _endasm
}
//
#pragma code
//
/** D E C L A R A T I O N S **************************************************/
//
//*****************************************************************************
// Keys pressed FIFO functions
//*****************************************************************************
int WriteRLEBuffer(int address, byte data)
{
	// access function for the split buffer in different RAM sections
	// returns the modulo address
	if(address<HALF_RLE_BUFFER_SIZE)
	{
	RLEBuffer[address]=data;
	} else
	{
	RLEBuffer2[(address-HALF_RLE_BUFFER_SIZE)]=data;
	}
	return (address);
}

int ReadRLEBuffer(int address, byte* data)
{
	// returns the data in *data and the address as the main return value
	if(address<HALF_RLE_BUFFER_SIZE)
	{
	*data=RLEBuffer[address];
	} else
	{
	*data=RLEBuffer2[(address-HALF_RLE_BUFFER_SIZE)];
	}
	return (address);
}

void incrementRLEPointer(unsigned int *ptr)
{
    if((*ptr)<(RLE_BUFFER_SIZE-1))(*ptr)++; else (*ptr)=0;
}

byte getRLE(void)
{
	byte g;
	if(RLEFull>0)
    	{
	    ReadRLEBuffer(RLEGetPtr, &g);
	    incrementRLEPointer(&RLEGetPtr);
	    RLEFull--;
	    } else g=0xFF;
	return g;
}

void putRLE(byte k)
{
    unsigned int x;
    double f;
     // overwrite mode!
	 if((RLEMode==0)&&(RLEFull>=RLE_BUFFER_SIZE))
	 {
  	 // the buffer is full, so we overwrite some data to make space for new data!
	 retrieveRLEData((byte*)&x, (byte*)&f);   // so we count the log time total off...
	 }
	
 	 // then write!
     if(RLEFull<RLE_BUFFER_SIZE)
     {
	 WriteRLEBuffer(RLEPtr, k);
	 incrementRLEPointer(&RLEPtr);
	 RLEFull++;
	 }
}    

void initRLE(void)
{
	byte i;
	//
	RLETotalTime=0.0;
    RLEGetPtr=0;
    RLEFull=0;   
    RLEPtr=0;
	for(i=0; i<RLE_CHANNELS;i++)
	{ 
	RLEMultiplicity[i]=0;
	RLEData[i]=0.0;
	}
}    

void clearRLE(void)
{
    byte interrupt;
    interrupt=INTCONbits.GIE;
    INTCONbits.GIE=0;
    initRLE();
    if(interrupt)INTCONbits.GIE=1;
}

void flushRLEData(byte channel)
{
	byte *ptr;
	byte *ptr2;
	byte x;
	unsigned int ch;

	channel&=RLE_CHANNEL_MASK;
	ch=(unsigned int)channel;
	
	if((RLEMode==0)||((RLEMode==1)&&(RLEFull<RLE_BUFFER_SIZE)))RLETotalTime+=(logPeriod*(double)RLEMultiplicity[channel]);
	
	RLEMultiplicity[channel]|=(ch<<RLE_BIT_SHIFT);
	ptr=(byte*)&RLEMultiplicity[channel];
	ptr2=(byte*)&RLEData[channel];
	putRLE(*ptr++);
	putRLE(*ptr);
	putRLE(*ptr2++);
	putRLE(*ptr2++);
	putRLE(*ptr2++);
	putRLE(*ptr2);
	RLEMultiplicity[channel]=0;
}

void flushAllRLEChannels(void)
{
	byte i;
	for(i=0; i<RLE_CHANNELS; i++)
	{
	flushRLEData(i);
	}
}

void retrieveRLEData(byte *multiplicityptr, byte* doubleptr)
{
    unsigned int mult;
    double data;
    byte* ptr1;
    byte* ptr2;
    
    ptr1=(byte*)&mult;
    ptr2=(byte*)&data;
    *ptr1=getRLE();
    *multiplicityptr=*ptr1;
    ptr1++;
    multiplicityptr++;
    *ptr1=getRLE();
    *multiplicityptr=*ptr1;
	*ptr2=getRLE();
	*doubleptr=*ptr2;
	ptr2++;
	doubleptr++;
	*ptr2=getRLE();
	*doubleptr=*ptr2;
	ptr2++;
	doubleptr++;
	*ptr2=getRLE();
	*doubleptr=*ptr2;
	ptr2++;
	doubleptr++;
	*ptr2=getRLE();
	*doubleptr=*ptr2;
	mult&=RLE_COMPLEMENT_MASK;
	if(mult<MAX_RLE_MULTIPLICITY)RLETotalTime-=(logPeriod*(double)mult);  // 0xFFFF is the empty virtual data popped when there is no data...
}

void popRLEData(byte* outptr)
{
	byte i;
	byte* outptr2;
	byte x;

	flushAllRLEChannels();
	outptr2=outptr;
	outptr++;
	i=0;
	while(i<16)
	{
	retrieveRLEData(outptr, outptr+2);
	x=*outptr;
	x&=(*(outptr+1));
	if(x==0xFF)break;
	outptr+=6;	
	i++;
	}
	*outptr2=i;
}

void pushRLEData(byte channel, double data, double percenterror, double absoluteerror)
{
	double epsilon;
	double delta;

	channel&=RLE_CHANNEL_MASK;
	if(RLEMultiplicity[channel]==0)
	{
		RLEMultiplicity[channel]++;
		RLEData[channel]=data;
	} else
	{
    	delta=(data-RLEData[channel]);
    	if(delta<0.0)delta=-delta;
    	epsilon=RLEData[channel]*percenterror/100.0;
    	if(epsilon<0.0)epsilon=-epsilon;
    	if(epsilon<absoluteerror)epsilon=absoluteerror;
    	 // now delta is the difference and epsilon is the maximum of the relative and absolute errors...
    	 if(delta<=epsilon)
		 {
		    if(RLEMultiplicity[channel]<MAX_RLE_MULTIPLICITY)RLEMultiplicity[channel]++;
		    else
		    flushRLEData(channel);
		 }
		 else
		 {
		 flushRLEData(channel);
		 }	  		
	}
}

// End of RLE Subsystem
void goToLowPowerMode(void)
{
	if(USBSENSE==0)
	{
	OSCCON=0x80;
	_asm
		sleep
	_endasm
	}
}

void delayMs(unsigned int delay)
{
	unsigned int n;
	//
 	while(delay--)
	{
	for(n = 0; n < 500; n++) 
	{
	// the following must be around a 2us delay...	
	_asm
		nop
		nop
		nop
		nop
	_endasm
	}	
 	}  
}

//*****************************************************************************
// Keys pressed FIFO functions
//*****************************************************************************
byte getKey(void)
{
	byte g;
	if(keyFull>0)
    	{
	    g=keyBuffer[keyGetPtr];
	    if(keyGetPtr<(KEY_BUFFER_SIZE-1))keyGetPtr++; else keyGetPtr=0;
	    keyFull--;
	    } else return 0xFF;
	return translateKeyCode(g);
}

/*
byte getNumberKey(void)
{
	byte g;
	if(keyFull>0)
    	{
	    g=keyBuffer[keyGetPtr];
	    keyGetPtr++;
	    if(keyGetPtr>=KEY_BUFFER_SIZE)keyGetPtr=0;
	    keyFull--;
	    } else return 0xFF;
	return translateNumberCode(g);
}
*/

void putKey(byte k)
{
	resetTimeOut();
	if(!standBy)setBrightnessLevel(brightnessPercent);
    if(keyFull<KEY_BUFFER_SIZE)
     {
	 keyBuffer[keyPtr]=k;
	 if(keyPtr<(KEY_BUFFER_SIZE-1))keyPtr++; else keyPtr=0;
	 keyFull++;
	 }
}    

void initKeys(void)
{
    keyGetPtr=0;
    keyFull=0;   
    keyPtr=0;
}    

byte swapBits(byte input)
{
	byte result;
	result=0;
	if(input&0x80)result|=0x10;
	if(input&0x40)result|=0x20;
	if(input&0x20)result|=0x40;
	if(input&0x10)result|=0x80;
	return result;
}

volatile void writeLCD(byte data, byte command)
{
		int i;
		byte firstdata;
		byte seconddata;
		byte interrupt;
	
		firstdata=swapBits(data);
		seconddata=swapBits(data<<4);
		LCDE=0;
		if(command==LCDCOMMAND)LCDRS=0; else LCDRS=1;
		PORTB=firstdata;
		interrupt=INTCONbits.GIE;
		INTCONbits.GIE=0;
		_asm
		    nop
		    nop
		_endasm
		LCDE=1;
		_asm
			nop
			nop
		_endasm
		LCDE=0;
		PORTB=seconddata;
		_asm
		    nop
		    nop
			nop
			nop
		_endasm
		LCDE=1;
		_asm
			nop
			nop
		_endasm
		LCDE=0;
		PORTB&=scanLn;
		_asm
		    nop
			nop
		    nop
		_endasm
		if(interrupt)INTCONbits.GIE=1;
		for(i=0; i<LCD_DELAY; i++)
		{	
			_asm
				nop
				nop
			_endasm
		}
}

void writeScreen(byte* inptr)
{
	byte i;
	writeLCD(0x80, LCDCOMMAND);
	for(i=0; i<(LCD_SIZE/2); i++)
	{
	writeLCD(*inptr, LCDDATA);
	inptr++;
	}
	writeLCD(0xC0, LCDCOMMAND);
	for(i=0; i<(LCD_SIZE/2); i++)
	{
	writeLCD(*inptr, LCDDATA);
	inptr++;
	}
}

void updateScreen(byte *inptr)
{
	if((!standBy)||(BUTTON==0)||((samplingCount & STANDBY_PERIOD)!=0))
	{
		writeScreen(inptr);
	}
}

void updateLCD(void)
{
	// function =  0 then update straight to the LCD screen...
	// function = -1 then update to buffer, then next update will shift left...
	// function = +1 then update to buffer, then next update will shift right... 
	
	int i;
	
	if(lcdTransitionType==0)
	{
		if(lastCall==0)
		{
		updateScreen((byte*)&lcdScreen[0]);
		} else
		{
		  for(i=0;i<(LCD_SIZE/2); i++)
		  {	
			updateScreen((byte*)&oldlcdScreen[0]);
			if(lastCall==1)
			{
			shiftGeneral(0, 15, lastCall, 2, (byte*)&oldlcdScreen[0]);
			oldlcdScreen[15]=lcdScreen[0];
			oldlcdScreen[31]=lcdScreen[16];
			shiftGeneral(0, 15, lastCall, 2, (byte*)&lcdScreen[0]);
			} else
			{
			shiftGeneral(0, 15, lastCall, 2, (byte*)&oldlcdScreen[0]);
			oldlcdScreen[0]=lcdScreen[15];
			oldlcdScreen[16]=lcdScreen[31];
			shiftGeneral(0, 15, lastCall, 2, (byte*)&lcdScreen[0]);
			}
		 	delayMs(SCROLL_DELAY/4);
 		  }
			updateScreen((byte*)&oldlcdScreen[0]);
			for(i=0; i<LCD_SIZE; i++)lcdScreen[i]=oldlcdScreen[i];
		}
		lcdTransitionType=0;
		lastCall=0;
	} else
	{
		for(i=0; i<LCD_SIZE; i++)oldlcdScreen[i]=lcdScreen[i];
		lastCall=lcdTransitionType;
		lcdTransitionType=0;
	} 
}

void clearLCD(void)
{
	int i;
	for(i=0; i<LCD_SIZE; i++)
	{
	lcdScreen[i]=' ';
	}
	updateLCD();
}

void initLCD(void)
{
		doBeep(160);				// pause 160ms
		PORTC=0;
		PORTB=0;
		writeLCD(0x28, LCDCOMMAND);
		writeLCD(0x28, LCDCOMMAND);
		delayMs(10);
		writeLCD(0x0C, LCDCOMMAND);
		delayMs(10);
		writeLCD(0x06, LCDCOMMAND);
		delayMs(10);
		writeLCD(0x01, LCDCOMMAND); // clear screen
		delayMs(150);
		clearLCD();
}

void shiftGeneral(int from, int to, int offset, int columns, byte* inptr)
{
	int i, j, k;

	j=0;
	k=0;
	while(j<columns)
	{
	if(offset>0)
	{
		i=from; 
		while(i<to)
		{
		*(inptr+i+k)=*(inptr+i+k+offset);
		i++;
		}
		*(inptr+to+k)=' ';
	} else
	if(offset<0)
	{
		i=to;
		while(i>from)
		{
		*(inptr+i+k)=*(inptr+i+k+offset);
		i--;
		}
		*(inptr+from+k)=' ';
	}
	k+=(LCD_SIZE/2);
	j++;
	}
}

/*
void shiftUpLCD(void)
{
	int i;
		
	for(i=0; i<(LCD_SIZE/2); i++)
	{
		lcdScreen[i]=lcdScreen[i+(LCD_SIZE/2)];
		lcdScreen[i+(LCD_SIZE/2)]=' ';
	}
}

void shiftDownLCD(void)
{
	int i;
		
	for(i=0; i<(LCD_SIZE/2); i++)
	{
		lcdScreen[i+(LCD_SIZE/2)]=lcdScreen[i];
		lcdScreen[i]=' ';
	}
}
*/

void shiftLeftLCD(void)
{
	shiftGeneral(0, 15, 1, 2, (byte*)&lcdScreen[0]);
}

void shiftRightLCD(void)
{
	shiftGeneral(0, 15, -1, 2, (byte*)&lcdScreen[0]);
}

/*
void leftTransitionLCD(void)
{
	int i, j;
	for(i=0; i<(LCD_SIZE/2); i++)
	{
		shiftLeftLCD();
		updateLCD();
		delayMs(SCROLL_DELAY/4);
	}
}

void rightTransitionLCD(void)
{
	int i, j;
	for(i=0; i<(LCD_SIZE/2); i++)
	{
		shiftRightLCD();
		updateLCD();
		delayMs(SCROLL_DELAY/4);
	}
}
*/

void writeStringLCD(byte cursor, rom char* instr)
{
		while((*instr)!='\0')
		{
			if(cursor<LCD_SIZE)lcdScreen[cursor++]=*instr;
			instr++;
		}
		//updateLCD();
}

void writeStringLCDDirect(byte cursor, rom char* instr)
{
		writeLCD(cursor, LCDCOMMAND);
		while((*instr)!='\0')
		{
			writeLCD(*instr, LCDDATA);
			instr++;
		}
}

void writeStringRamLCD(byte cursor, ram char* instr, int padding)
{
		while((*instr)!='\0')
		{
			if(cursor<LCD_SIZE)lcdScreen[cursor++]=*instr;
			instr++;
			padding--;
		}
		while(padding>0)
		{
			if(cursor<LCD_SIZE)lcdScreen[cursor++]=' ';
			padding--;
		}
		//updateLCD();
}

/*
byte disFix(byte in)
{
	in&=0x0F;
	if(in>9)in+=7;
	return in+0x30;
}
*/

int printITOA(unsigned long xx, int blanking, int decimal, int mode)
{
		// unsigned integers from 0 to 65535
		// blanking=0 implies zero blanking
		// decimal=   number of digits before the decimal to be printed if 0 then do not print a decimal point (blanking stops after the decimal)
		int i, j;
		unsigned long ux;
		char c;
		
		ux=(unsigned long)xx;
		j=NUM_DIGITS+1;
		for(i=NUM_DIGITS; i>=0; i--)
		{
		
		c=(char)(0x30+(0x0F&(ux%10)));
		ux=ux/10;
		if(j==decimal){ buffer[j--]='.'; }
		buffer[j--]=c;
		}
		buffer[NUM_DIGITS+2]=0;
		j=j+1;
		if(blanking==0)while((buffer[j]==0x30)&&(j<NUM_DIGITS))j++;
		if(mode==0)if((j>0)&&(buffer[j]=='.'))j--;
		return j;
}

rom double MultiplierF[11]={ 	1.0, 
								10.0, 
								100.0, 
								1000.0, 
								10000.0, 
								100000.0, 
								1000000.0, 
								10000000.0, 
								100000000.0, 
								1000000000.0, 
								10000000000.0 
};

int printFDecimal(double f, int numdec, int mode)
{
		if(f<0.0)f=-f;
		if(numdec<0)numdec=0; else if(numdec>10)numdec=10;
		f*=MultiplierF[numdec];
		f+=0.5;
		return printITOA(f, 0, (NUM_DIGITS+1)-numdec, mode);
}

int printFSignedDecimal(double f, int numdec, int mode)
{
		byte k;
		char c;

		if(numdec<0)numdec=0; else if(numdec>6)numdec=6;
		if(f<0){ f=-f; c='-'; } else c='+';
		f*=MultiplierF[numdec];
		f+=0.5;
		k=printITOA(f, 0, (NUM_DIGITS+1)-numdec, mode);
		k--;
		buffer[k]=c;
		return k;
}


void disFUnsignedLCD(byte cursor, double f, int numdec, int mode, int padding)
{
	int i;

	if(f<0.0)f=-f;
	i=printFDecimal(f, numdec, mode);
	writeStringRamLCD(cursor+1, (ram char*)&buffer[i], padding);
	updateLCD();
}

void disFLCD(byte cursor, double f, int numdec, int mode, int padding)
{
	int i;

	if(f<0.0)
	{
	f=-f;
	if(cursor<LCD_SIZE)lcdScreen[cursor]='-';
	} else
	{
	if(cursor<LCD_SIZE)lcdScreen[cursor]='+';
	}
	disFUnsignedLCD(cursor, f, numdec, mode, padding);
}

/*
void disaLCD(byte cursor, byte a)
{
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a>>4);
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a);
		updateLCD();
}
*/
/*
void diswordLCD(byte cursor, unsigned int a)
{
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a>>12);
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a>>8);
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a>>4);
		if(cursor<LCD_SIZE)lcdScreen[cursor++]=disFix(a);
		updateLCD();
}
*/

void initPowerSystem(void)
{
	POWERONTRIS=0;			// make it an output!
	POWERON=0;				// turn power on!
}

void updateRelayState(void)
{
	if(relayEnable)
	{
		if(relayOn)RELAYON=0; else RELAYON=1;
	} else RELAYON=1;	// always off
}

void initRelaySystem(void)
{
	RELAYONTRIS=0;			// make it an output!
	RELAYON=1;				// turn relay off first of all...
}

byte usbSense(void)
{
    byte x, i;
    x=3;
    i=0;
    if(USBSENSE)
	{
		if(lastUSB!=1)
		{
		USBConnect();
		yesNoUSBString[0]='Y';
		yesNoUSBString[1]='e';
		yesNoUSBString[2]='s';
		i=1;
		lastUSB=1;
		}
	} else
	{
		if(lastUSB!=0)
		{
		yesNoUSBString[0]='N';
		yesNoUSBString[1]='o';
		yesNoUSBString[2]=' ';
		UIE=0;
		UCON=0;
		usb_device_state=0;
		i=1;
		lastUSB=0;
		}
	}
	if(i)
	{
	yesNoUSBString[x++]=' ';
	yesNoUSBString[x++]='(';
	yesNoUSBString[x++]='-';
	yesNoUSBString[x++]=')';
	yesNoUSBString[x++]='\0';
	} else
	yesNoUSBString[5]=0x30+usb_device_state;
    return lastUSB;
}

void powerOff(void)
{
	if(USBSENSE==0)
	{
		POWERON=1;
		OSCCON=0;
		_asm
			sleep
		_endasm
	}
}

void goOutOfStandBy(void)
{
	//if(USBSENSE==0)USBConnect();
	PIE2bits.TMR3IE=1;
	standBy=0;
}

void goInToStandBy(void)
{
	PIE2bits.TMR3IE=0;
	//if(USBSENSE==0)
	//{
	//UIE=0;
	//UCON=0;
	//}
	standBy=1;
}

void checkStandBy(void)
{
	double f, g, h;

	// a small amount of hysteresis is factored into the going in and out of standby mode...
	if(enableStandByDetect)
	{
	 f=chargeCurrent;			
	 if(f<=0.0)f=-f;
	 g=loadCurrent;
	 if(g<=0.0)g=-g;
	 if(standBy)
	 {
		h=currentThreshold*1.5;
		if((BUTTON==0)||(USBSENSE)||(f>=h)||(g>=h))
		{
			goOutOfStandBy();
		}
	 } else
	 {
		if((USBSENSE==0)&&(f<currentThreshold)&&(g<currentThreshold))
		{
			goInToStandBy();
		}
	 }
	} else standBy=0;
}

rom byte keyCodes[16]   ={ 'A', '3', '2', '1', 'B', '6', '5', '4', 'C', '9', '8', '7', 'D', '#', '0', '*' };

byte translateKeyCode(byte rawCode)
{
	return keyCodes[rawCode];
}

double enterNumber(byte cursor, int numdec, int maxchars, double defaultValue)
{
	double f;
	ram char string[16];
	if(maxchars>15)maxchars=15;
	enterNumberString(cursor, &string[0], maxchars);
	if(string[0]=='\0')
	{
	f=defaultValue;
	}
	else
	{
	f=atof(&string[0]);
	}
	disFLCD(cursor, f, numdec, 0, maxchars);
	return f;
}

void enterNumberString(byte cursor, ram char* string, int maxchars)
{
	byte key, index, i;

	key=0;
	index=0;
  	while(key!='#')
	{
	while(keyFull==0);
	key=getKey();
	if(key=='A')key='+'; else if(key=='B')key='-'; else if(key=='*')key='.';
		if((key!='D')&&(key!='#'))
		{
			if(index<maxchars)
			{
			lcdScreen[index+cursor]=key;  
			updateLCD();
			string[index]=key;
			}
			if(index<maxchars)index++;
		} else
		if(key=='D')
		{ 
			if(index>0)index--; 
			i=index;
			while(i<maxchars)
			{
			lcdScreen[i+cursor]=' ';
			updateLCD();
			string[i]=' ';
			i++;
			}
		} else
		if(key=='#')
		{
			string[index]='\0';
		}
	}
}

void initKeypad(void)
{
	scanCounter=0;
	scanCode=0;
	scanLn=0;
	T3CON=0b10100001;		// 1:8 prescaler from system clock synchronised , 16 bit operation
//	PIR2bits.TMR3IF=0;
	IPR2bits.TMR3IP=0;		// low priority interrupt
	PIE2bits.TMR3IE=1;		// enable interrupt
	INTCON2bits.RBPU=0;		// enable weak pull ups on port B
//	INTCONbits.RBIF=0;
//	INTCON2bits.RBIP=0;		// low priority interrupt
//	INTCONbits.RBIE=0;		// enable interrupt on change interrupt
}

void initSampling(void)
{
	// initialize the sampling system
	// which samples the 5 ADC readings
	// periodically in the background
	int i;

	samplingIndex=0;
	samplingCount=0;
	for(i=0; i<(NUM_SAMPLES-2); i++)
	{
	sample[i].value=0.0;
	sample[i].avgCount=0;
	sample[i].avgValue=0.0;	
	sample[i].totalValue=0.0;
	sample[i].updated=0;
	}	
	capacityAs=DEFAULT_CAPACITY_AS;
	time=0.0;
	itime=0.0;
	T1CON=0;				// turn off
	T1CON=0x81;				// 1:1 prescaler from system clock synchronised, 16 bit operation
//	PIR1bits.TMR1IF=0;
	IPR1bits.TMR1IP=1;		// high interrupt priority
	PIE1bits.TMR1IE=1;		// enable interrupt
}

int initADC(void)
{
	ADCON1=0x0B;			// enable AN0 to AN3 as analog inputs the rest as digital pins
	ADCON2=0xBE;			// right justified, FOSC/16 (should not be lower than this, ie faster) and 20 TAD
	ADCON0=0x01;
}

double readAN(byte channel)
{
	unsigned int i;
	byte x;
	// returns a number from 0.0 to 1023.0 (or a bit higher depending on ref voltage)
	ADCON0&=0xC3;
	i=0x3C & ((channel & 0x0F)<<2);
	ADCON0|=i;			// set the channel
	i=0;
	ADCON0|=0x02;
	while((ADCON0 & 0x02)!=0);
	return (double)((((unsigned int)ADRESH)<<8) + ADRESL);
}

void initDelayTimer(void)
{
	timerDelay=0;
	INTCONbits.TMR0IF=0;
	INTCON2bits.TMR0IP=0;	// interrupt priority
	T0CON=0x87;				// 1:256 prescaler from instruction clock at 12000000Hz, gives around 1400ms ticks
	INTCONbits.TMR0IE=1;
}

void initPWM(void)
{
	// initialize the PWN on CCP1
	// CCP1CON=0x00;
	TMR2=0x00;
	PR2=0xFF;
	//PIR1bits.TMR2IF=0;			// clear ack
	//PIE1bits.TMR2IE=1;			// do not enable interrupt
	//IPR1bits.TMR2IP=0;			// low priority
	T2CON=0x7F;					
	TRISC&=0xFB;
	//	CCPR1L=0;
	//	CCPR1H=0;
	TMR2=0xFE;					// note that CCP1RH is not latched in unitl there's a match between TMR2 and PR2 ie the cycle starts again.
	//	PIR1bits.CCP1IF=0;
	//	PIE1bits.CCP1IE=0;
	//	IPR1bits.CCP1IP=0;
	CCP1CON=0x0C;				// enable PWM
}

int putPWM(byte xx)
{
	// only 8 bits are used of xx 
	brightness=(0xFE & xx);
	CCPR1L=(brightness);
}

void USBConnect(void)
{
	mInitializeUSBDriver();     // See usbdrv.h
 	USBCheckBusStatus();        // Modified to always enable USB module, M.G.
}

void initBeep(void)
{
	BEEP=0;
}

void doBeep(int delay)
{
	if(beepOn)
	{
		PIE2bits.TMR3IE=0;
		alarmOn|=2;
		BEEP=1;
		delayMs(delay);
		BEEP=0;
		alarmOn&=~2;
		PIE2bits.TMR3IE=1;
	} else BEEP=0;
}

void idF(int i)
{

}

ram char* copyString(ram char* instr, ram char* outstr, char delimiter)
{
	while((*instr)!=delimiter)
	{
	*outstr++=*instr++;
	}
	return outstr;
}

void updateTimeString(double floatminutes)
{
	// update the time string with minutes in double...
	ram char* outstr;
	long days;
	int hours;
	int minutes;
	byte k;

	if(floatminutes<0.0)floatminutes=0.0;
	days=(long)(floatminutes/1440.0);
	floatminutes-=(days*1440.0);
	hours=(int)(floatminutes/60.0);
	floatminutes-=(hours*60.0);
	minutes=(int)floatminutes;
	outstr=&timeString[0];
	k=printFDecimal(days, 0, 0);
	outstr=copyString(&buffer[k], outstr, '.');
	*outstr++='d';
	*outstr++=' ';
	k=printFDecimal(hours,0,0);
	outstr=copyString(&buffer[k], outstr, '.');
	*outstr++='h';
	*outstr++=' ';
	k=printFDecimal(minutes,0,0);
	outstr=copyString(&buffer[k], outstr, '.');
	*outstr++='m';
	*outstr='\0';
}

void clearWithoutUpdate(void)
{
	int i;
	for(i=0; i<32; i++)lcdScreen[i]=' ';
}

void clearBottomLine(void)
{
	int i;
	for(i=16; i<32; i++)lcdScreen[i]=' ';
	updateLCD();
}

void flushAndWaitForKey(void)
{
	initKeys();
	while(keyFull==0);
}

void writeTitleStringMenu(rom char* instr, double min, double max, byte numdec)
{
		clearBottomLine();
		disFUnsignedLCD(16, min, numdec, 0, 5);
		lcdScreen[22]='-';
		disFUnsignedLCD(23, max, numdec, 0, 5);
		writeStringLCD(0, instr);
		updateLCD();
}

void doEnterNumberMenu(rom char* instr, double min, double max, byte numdec, double* result)
{
	byte i;
	i=0;
	while((i==0)||((*result)>max)||((*result)<min))
	{
	i=1;
	writeTitleStringMenu(instr, min, max, numdec);
	flushAndWaitForKey();
	clearBottomLine();
	(*result)=enterNumber(16, numdec, 16, *result);
	}
	delayMs(1000);
	clearLCD();
	writeStringLCD(0, (rom char*)"Entered:");
	disFLCD(16, *result, numdec, 0, 16);
	delayMs(1000);
}

void printPercentageBar(byte from, byte to, double percentage)
{
	byte i, threshold;
	double f;

    f=percentage;
	if(f<0.0)f=0.0; else if(f>1.0)f=1.0;
	i=from;
	threshold=(byte)((to+1)*f+((from)*(1.0-f)));
	while(i<=to)
	{
	if(i<threshold)lcdScreen[i]=0xFF ; else lcdScreen[i]=' ';
	i++;
	}
}

void doEnterValueUpAndDownMenu(rom char* instr, double* value, byte numdec, void (*setFunction)(double), double step, double min, double max)
{
	char keyPressed;

	clearLCD();
	writeStringLCD(0, instr);
	updateLCD();
	keyPressed=0;
	while(keyPressed!='#')
	{
	setFunction(*value);
	disFLCD(27, *value, numdec, 0, 5);
	lcdScreen[16]='[';
	lcdScreen[26]=']';
	printPercentageBar(17, 25, (((*value)-min)/(max-min)));
	updateLCD();
		while(keyFull==0);
		keyPressed=getKey();
		switch(keyPressed)
		{
			case 'A':
				*value+=step;
				break;
			case 'B':
				*value-=step;
				break;
			default:
				break;
		}
	}
}

void seePercentageFull(rom char* instr, double *value, int* valueint, double max, byte startNum, byte numdec)
{
    double f;
    char keyPressed;
    
    if(startNum<19)startNum=19;
	clearLCD();
	writeStringLCD(0, instr);
	updateLCD();
	keyPressed=0;
    while(keyPressed!='#')
    {	
    if(value==0)f=(double)(*valueint); else f=*value;
    disFLCD(startNum, 100.0*f/max, numdec, 0, 5);
	lcdScreen[16]='[';
	lcdScreen[(startNum-1)]=']';
	printPercentageBar(17, startNum-2, f/max);
	updateLCD();
	delayMs(SCROLL_DELAY);
	keyPressed=getKey();
	}
}    

void doEnterOptionsMenu(rom char* instr, byte* valueIndex, ram char* displayString, void (*setFunction)(byte), byte modulus)
{
	char keyPressed;

	clearLCD();
	writeStringLCD(0, instr);
	updateLCD();
	keyPressed=0;
	while(keyPressed!='#')
	{
	setFunction(*valueIndex);
	lcdScreen[16]=(char)(0x30+(*valueIndex));
	lcdScreen[17]=':';
	writeStringRamLCD(19, displayString, 13);
	updateLCD();
		while(keyFull==0);
		keyPressed=getKey();
		switch(keyPressed)
		{
			case 'A':
				if((*valueIndex)>0)(*valueIndex)--;
				break;
			case 'B':
				if((*valueIndex)<(modulus-1))(*valueIndex)++;
				break;
			default:
				break;
		}
	}
}

void displaySettingMenu(rom char* instr, byte numdec, double* result, byte signedDisplay, ram char* valueString, int blocking)
{
	byte key;
	if(blocking==1)initKeys();				// flush queue
	key=0;
	while(key!='#')
	{
	clearWithoutUpdate();
	writeStringLCD(0, instr);
	if(valueString==0)
	{
	if(signedDisplay)disFLCD(16, *result, numdec, 0, 16); else disFUnsignedLCD(16, *result, numdec, 0, 16);
	} else
	{
	writeStringRamLCD(17, valueString, 15);
	updateLCD();
	}
		if(blocking==1)
		{
		delayMs(SCROLL_DELAY*2);
		key=getKey();
		initKeys();
		} 
		else break;
	}
}

int romStringLength(rom char *instr)
{
	// returns the length of the string in rom
	int i;
	i=0;
	while((*instr)!='\0')
	{
	i++;
	instr++;
	}
	return i;
}

void displayTwoSettingsMenu(rom char* instr1, byte numdec1, double* result1, byte signedDisplay1, ram char* valueString1, rom char* instr2, byte numdec2, double* result2, byte signedDisplay2, ram char* valueString2, int blocking)
{
	byte key, offset;
	if(blocking==1)initKeys();				// flush queue
	key=0;
	while(key!='#')
	{
		clearWithoutUpdate();
		writeStringLCD(0, instr1);
		offset=romStringLength(instr1);
		if(valueString1==0)
		{
			if(signedDisplay1)disFLCD(offset, *result1, numdec1, 0, 16); else disFUnsignedLCD(offset, *result1, numdec1, 0, 16);
		} 
		else
		{
			writeStringRamLCD(offset, valueString1, 16);
			updateLCD();
		}
		writeStringLCD(16, instr2);
		offset=romStringLength(instr2);
		if(valueString2==0)
		{
			if(signedDisplay1)disFLCD(16+offset, *result2, numdec2, 0, 16); else disFUnsignedLCD(offset+16, *result2, numdec2, 0, 16);
		} 
		else
		{
			writeStringRamLCD(offset+16, valueString2, 16);
			updateLCD();
		}
		if(blocking==1)
		{
		delayMs((SCROLL_DELAY*2));
		key=getKey();
		initKeys();
		} 
		else 
			break;
	}
}

void seeBatteryChargeAndMinutes(int i)
{
	updateTimeString(batteryMinutes);
	displayTwoSettingsMenu(	(rom char*)"Chrg.(%): ", 1, 		(double*)&batteryPercent, 1, 0,
							(rom char*)"Cycle(s): ", 0, (double*)&cycles, 0, 0,
							i);
}

void seeBatteryVoltageCurrent(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Volts: ", 1, (double*)&batteryVoltage, 1, 0,
							(rom char*)"Amps : ", 3, (double*)&netCurrent, 1, 0,
							i);
}

void seeChargeCurrentAndLoadCurrent(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Chrg(A): ", 3, (double*)&chargeCurrent, 1, 0,
							(rom char*)"Load(A): ", 3, (double*)&loadCurrent, 1, 0,
							i);
}

void seeChargeAndVoltage(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Chrg.(%): ", 1, (double*)&batteryPercent, 1, 0,
							(rom char*)"Volt.(V): ", 1, (double*)&batteryVoltage, 1, 0,
							i);
}

void seeCapacityDouble(int i)
{
	displayTwoSettingsMenu(	(rom char*)"AH: ", 1, (double*)&batteryCapacityAH, 1, 0,
							(rom char*)"Load(A): ", 3, (double*)&loadCurrent, 1, 0,
							i);
}

void seeCircuit(int i)
{
	displayTwoSettingsMenu(	(rom char*)"Circt(mA): ", 0, (double*)&circuitCurrent, 1, 0,
							(rom char*)"Relay(mA): ", 3, (double*)&relayCurrent, 1, 0,
							i);
}

void seeWatts(int i)
{
	displayTwoSettingsMenu(	(rom char*)"WH: ", 1, (double*)&wattHours, 1, 0,
							(rom char*)"Load(W): ", 1, (double*)&loadWatts, 1, 0,
							i);
}

/*
void seeAllScrolling(int i)
{
	clearWithoutUpdate();
	writeStringLCD(0, (rom char*)"All Scrolling");
	updateLCD();
}
*/

void setAvgCount(int i)
{
	double f;
	f=avgSampleCount;
	doEnterNumberMenu((rom char*) "Average Samples?", 0.0, 120.0 , 0, (double*)&f);
	avgSampleCount=(double)((byte)(f+0.5));
}

void setCutOffVoltage(int i)
{
	doEnterNumberMenu((rom char*) "Shutdown Volt.?", 0.0, 60.0 , 1, (double*)&cutOffVoltage);
}

void setBrightnessPercent(int i)
{
	doEnterValueUpAndDownMenu((rom char*)"Brightness (%)? ", (double*)&brightnessPercent, 0, setBrightnessLevel, 5.0, 0.0, 100.0);
}

void setAlarmInternal(int i)
{
	if(alarmCapacity<-60.0)alarmCapacity=-60.0; else if(alarmCapacity>100.0)alarmCapacity=100.0;
}

void setAlarm(int i)
{
	doEnterValueUpAndDownMenu((rom char*)"Alarm (%)? ", (double*)&alarmCapacity, 0, setAlarmInternal, 1.25, -60.0, 100.0);
	//doEnterNumberMenu((rom char*)"Alarm Below (%)?", 0.0, 100.0, 1, (double*)&alarmCapacity);
}

void setTrickle(int i)
{
	doEnterNumberMenu((rom char*) "Trickle (A)?", 0.0, 99.0, 3, (double*)&trickleCurrent);
}

void setThreshold(int i)
{
	doEnterNumberMenu((rom char*) "Threshold (A)?", 0.0, 99.0, 3, (double*)&currentThreshold);
}

/*
void seeRLEBuffer(int i)
{
    seePercentageFull((rom char*) "RLE Buffer (%)", 0, (int*)&RLEFull, (double)RLE_BUFFER_SIZE, 27, 0);
}
*/

void seeRLEPosition(int i)
{
    double f;
    char keyPressed;
    byte from, full;
    byte t, u;
    
 	clearLCD();
	if(RLEMode==0)writeStringLCD(0, (rom char*)"RLE Pos. (Bytes)"); else writeStringLCD(0, (rom char*)"RLE Buffer (%)");
	updateLCD();
	keyPressed=0;
    while(keyPressed!='#')
    {	
    f=(double)RLEGetPtr/((double)(RLE_BUFFER_SIZE-1));
    from=(byte)((f*10.0)+16.0);
    f=(double)RLEFull/((double)(RLE_BUFFER_SIZE-1));
    full=(byte)(f*9.0);
    if(RLEMode==0)disFLCD(27, RLEFull, 0, 0, 5); else disFLCD(27, 100.0*(double)RLEFull/(double)RLE_BUFFER_SIZE, 0, 0, 5);
	t=16;
	while(t<=26)lcdScreen[t++]=' ';
	lcdScreen[from]='[';	
    t=from+1;
    while(t<(from+full))
    {
    if(t>26)u=t-11; else u=t;
    lcdScreen[u]=0xFF;
    t++;
    }
    if(t>26)u=t-11; else u=t;
    lcdScreen[u]=']';
	updateLCD();
	delayMs(SCROLL_DELAY);
	keyPressed=getKey();
	}
}
    
void setCheckPeriod(int i)
{
    double f;
    f=checkPeriod;
	doEnterNumberMenu((rom char*) "Check Prd.(sec)?", 5.0, 86400.0, 0, (double*)&f);
	checkPeriod=(double)((unsigned int)(f+0.5));	
}

void setVoltageRail50(int i)
{
	doEnterNumberMenu((rom char*) "+5V Rail (V)?", 4.0, 6.0, 2, (double*)&voltageRail50);
}

/*
void calibrateVoltageRail50(int i)
{
	double g;
	if(areYouSure())
	{
	while(totalUpdated==0);
	g=3.30000/vusbRail;
	voltageRail50*=g;
	}
}
*/

/*
void setVoltageRail33(int i)
{
	doEnterNumberMenu((rom char*) "+3.3V Rail (V)? ", 3.0, 3.6, 2, &voltageRail33);
}
*/

void setFullCapacity(int i)
{	
	doEnterNumberMenu((rom char*) "Capacity (AH)?", 0.0, 999999.0, 1, (double*)&fullCapacity);
}

void setLoadDifferentialGain(int i)
{	
	doEnterNumberMenu((rom char*) "Load Amp. Gain?", 1.0, 1000.0, 2, (double*)&loadDifferentialGain);
}

void setChargeDifferentialGain(int i)
{	
	doEnterNumberMenu((rom char*) "Chrg. Amp. Gain?", 1.0, 1000.0, 2, (double*)&chargeDifferentialGain);
}

void setSenseResistance(int i)
{	
	doEnterNumberMenu((rom char*) "Sense Res.(\xF4)?", 0.1, 100.0, 1, (double*)&senseResistance);
}

void setShuntResistance(int i)
{	
	doEnterNumberMenu((rom char*) "Shunt Res.(m\xF4)?", 0.1, 9999.0, 3, (double*)&shuntResistance);
}


double calibrateDivider(rom char *instr, byte index, double defaultValue)
{
	double f, g;
	byte y;

	y=CCP1CON;
	CCP1CON=0;
	PORTCbits.RC2=1;			// turn LCD on
	f=defaultValue;
	doEnterNumberMenu(instr, 0.0, 60.0, 2, &f);
	g=(readAN(index)*voltageRail50/1023.0);
	CCP1CON=y;
	if(g>0.0)return f/g;
	return 1.0;
}

void setDividerLow(int i)
{
	dividerLow=calibrateDivider((rom char*) "Lo. Voltage Now?", ANCH_VBATTNEG, circuitVoltage);
}

void setDividerHigh(int i)
{
	dividerHigh=calibrateDivider((rom char*)"Hi. Voltage Now?", ANCH_VBATTPOS, batteryVoltage);
}

void calibrateCurrentSystem(int i)
{	
	double f, v1, i1, t, w;
	byte y;
	f=0.0;
	doEnterNumberMenu((rom char*)"Load Now (A)?", 0.0, 1000.0, 3, &f);
	if(f>0.0)
	{
		y=CCP1CON;
		CCP1CON=0;
		PORTCbits.RC2=1;		// turn on LCD 
		delayMs(1000);			// wait!
		while(totalUpdated==0);
		i1=circuitCurrent;
		v1=loadDiffAmpVoltage;
		PORTCbits.RC2=0;		// turn off LCD 
		delayMs(1000);			// wait!
		while(totalUpdated==0);
		i1-=circuitCurrent;
		v1-=loadDiffAmpVoltage;
		w=loadCurrent;
		delayMs(1000);
		CCP1CON=y;
		if(i1>1.0)
		{
		t=((-v1)*1000.0/(loadDifferentialGain*i1));
		}	// no else here please!
		if(t<=0.001)t=rX;
		rX=t;
		shuntResistance*=(w/f);
	}
}

void detectSenseResistance(int i)
{
     double f;
     byte y;
     
     f=0.0;   
     y=CCP1CON;
     CCP1CON=0;
     PORTCbits.RC2=1;           // turn on LCD
     doEnterNumberMenu((rom char*)"Circt. Now (mA)?", 0.0, 1000.0, 3, &f);
     if(f>0.0)
     {
        // f is now the circuit current in mA
        while(totalUpdated==0);             // wait!
        senseResistance=(1000.0*(batteryVoltage-circuitVoltage))/f;
	 }
	 CCP1CON=y;
}

void setChargingEfficiency(int i)
{	
	doEnterNumberMenu((rom char*) "Efficiency (%)?", 0.1, 100.0, 1, (double*)&chargingEfficiency);
}

void setCyclePercent(int i)
{	
	doEnterNumberMenu((rom char*) "Cycle Thr. (%)?", 10.0, 40.0, 1, (double*)&cyclePercent);
	cycles=0.0;
}

void setRelayCoilResistance(int i)
{	
	doEnterNumberMenu((rom char*) "Relay Coil (\xF4)?", 1.0, 99999.0, 1, (double*)&relayCoilResistance);
}

void seeFirmwareVersion(int i)
{
	displaySettingMenu((rom char*)"Version", 2, (double*)&firmwareVersion, 0, 0, i);
}

void seeRx(int i)
{
	displaySettingMenu((rom char*)"Internal Res.(\xF4)", 6, (double*)&rX, 0, 0, i);
}

void seeBatteryPercent(int i)
{
	displaySettingMenu((rom char*)"Charge (%)", 1, (double*)&batteryPercent, 1, 0, i);	// signed
}

void seeCycles(int i)
{
	displaySettingMenu((rom char*)"Cycles", 0, (double*)&cycles, 0, 0, i);
}

void seeBatteryCapacityAH(int i)
{
	displaySettingMenu((rom char*)"Capacity (AH)", 3, (double*)&batteryCapacityAH, 1, 0, i); // signed
}

void seeBatteryVoltage(int i)
{
	displaySettingMenu((rom char*)"Voltage (V)", 1, (double*)&batteryVoltage, 0, 0, i);
}

void seeVoltageRail50(int i)
{
	displaySettingMenu((rom char*)"+5V Rail (V)", 2, (double*)&voltageRail50, 0, 0, i);
}

/*
void seeVUSBRail(int i)
{
	displaySettingMenu((rom char*)"+3.3V Rail (V)", 2, (double*)&vusbRail, 0, 0, i);
}
*/

void seeCircuitVoltage(int i)
{
	displaySettingMenu((rom char*)"Circuit (V)", 1, (double*)&circuitVoltage, 0, 0, i);
}

void seeLoadDiffAmpVoltage(int i)
{
	displaySettingMenu((rom char*)"Load Amp (V)", 3, (double*)&loadDiffAmpVoltage, 0, 0, i);
}

void seeChargeDiffAmpVoltage(int i)
{
	displaySettingMenu((rom char*)"Charge Amp (V)", 3, (double*)&chargeDiffAmpVoltage, 0, 0, i);
}

void seeLoadCurrent(int i)
{
	displaySettingMenu((rom char*)"Load (A)", 3, (double*)&loadCurrent, 0, 0, i);
}

void seeChargeCurrent(int i)
{
	displaySettingMenu((rom char*)"Charge (A)", 3, (double*)&chargeCurrent, 0, 0, i);
}

void seeCircuitCurrent(int i)
{
	displaySettingMenu((rom char*)"Circuit (mA)", 0, (double*)&circuitCurrent, 0, 0, i);
}

void seeNetBatteryDrain(int i)
{
	displaySettingMenu((rom char*)"Net (A)", 3, (double*)&netCurrent, 1, 0, i);	// signed
}

void seeBatteryTimeMinutes(int i)
{
    char keyPressed;
    
    keyPressed=0;
    while(keyPressed!='#')
    {
    while(totalUpdated==0);
	updateTimeString(batteryMinutes);
	displaySettingMenu((rom char*)"Time", 0, 0, 0, (char*)&timeString[0], 0);   // non blocking!
	 if(i==0)break;
	 else
	 {
    	keyPressed=getKey();
	    initKeys();
	    delayMs(SCROLL_DELAY);
	 }    
	}
}

void seeFuseString(int i)
{
    char keyPressed;
    
    keyPressed=0;
    while(keyPressed!='#')
    {
    while(totalUpdated==0);
	writeFuseString(currentFuse);
	displaySettingMenu((rom char*)"Overload Fuse", 0, 0, 0, (char*)&fuseString[0], 0);   // non blocking!
	 if(i==0)break;
	 else
	 {
    	keyPressed=getKey();
	    initKeys();
	    delayMs(SCROLL_DELAY);
	 }    
	}
}

void seeLastSync(int i)
{
    char keyPressed;
    
    keyPressed=0;
    while(keyPressed!='#')
    {
    if(lastSync>=0.0)updateTimeString(lastSync);
	else
	{
	timeString[0]='N';
	timeString[1]='/';
	timeString[2]='A';
	timeString[3]='\0';
	}
	displaySettingMenu((rom char*)"Last Sync. (ago)", 0, 0, 0, (char*)&timeString[0], 0);   // non blocking!
	 if(i==0)break;
	 else
	 { 
    	    keyPressed=getKey();
    	    initKeys();
    	    delayMs(SCROLL_DELAY);
     }  	    
    }
}

void seeLogTime(int i)
{
    char keyPressed;
    
    keyPressed=0;
    while(keyPressed!='#')
    {
    updateTimeString(RLETime/60.0);
	displaySettingMenu((rom char*)"Last Sample(ago)", 0, 0, 0, (char*)&timeString[0], 0);   // non blocking!
	 if(i==0)break;
	 else
	 { 
    	    keyPressed=getKey();
    	    initKeys();
    	    delayMs(SCROLL_DELAY);
     }  	    
    }
}

void seeLogTotalTime(int i)
{
    char keyPressed;
    
    keyPressed=0;
    while(keyPressed!='#')
    {
    updateTimeString(RLETotalTime/60.0);
	displaySettingMenu((rom char*)"Log Total Time", 0, 0, 0, (char*)&timeString[0], 0);   // non blocking!
	 if(i==0)break;
	 else
	 { 
    	    keyPressed=getKey();
    	    initKeys();
    	    delayMs(SCROLL_DELAY);
     }  	    
    }
}

void seeUSBStatus(int i)
{
	displaySettingMenu((rom char*)"USB Connected", 0, 0, 0, (char*)&yesNoUSBString[0], i);
}

void updateBeeperString(byte x)
{
	beepOn=(x & 1);
	if(beepOn)
	{
		yesNoBeeperString[0]='O';
		yesNoBeeperString[1]='n';
		yesNoBeeperString[2]='\0';
	}
	else
	{
		yesNoBeeperString[0]='O';
		yesNoBeeperString[1]='f';
		yesNoBeeperString[2]='f';
		yesNoBeeperString[3]='\0';
	}
}

void updateRelayString(byte x)
{
	relayEnable=(x & 1);
	if(relayEnable)
	{
		relayString[0]='O';
		relayString[1]='n';
		relayString[2]='\0';
	}
	else
	{
		relayString[0]='O';
		relayString[1]='f';
		relayString[2]='f';
		relayString[3]='\0';
	}
}

void setBeeperStatus(int i)
{
	doEnterOptionsMenu((rom char*)"Beeper?", (byte*)&beepOn, (char*)&yesNoBeeperString[0], (char*)&updateBeeperString, 2);
}

void setPeukerts(int i)
{	
	doEnterNumberMenu((rom char*) "Peukert's?", 1.000, 2.000, 3, (double*)&peukertsConstant);
}


void setLogFloor(int i)
{	
	doEnterNumberMenu((rom char*) "Min. Abs. Error?", 0.0, 100.0, 3, (double*)&logFloor);
}

void setRelayHysteresis(int i)
{	
	doEnterNumberMenu((rom char*) "Hysteresis (V)?", 0.0, 10.0, 2, (double*)&relayHysteresis);
}

void setRelayOffVoltage(int i)
{	
	doEnterNumberMenu((rom char*) "Cut Off (V)?", 0.0, 60.0, 2, (double*)&relayOffVoltage);
}

void setLoadCurrentMax(int i)
{	
	doEnterNumberMenu((rom char*) "Max. Load (A)?", 0.0, 1000.0, 3, (double*)&loadCurrentMax);
}

void setCurrentMaxTimeOutPeriod(int i)
{	
	doEnterNumberMenu((rom char*) "Duration (sec)?", 0.0, 3600.0, 1, (double*)&currentMaxTimeOutPeriod);
}

void setRelayEnable(int i)
{
	doEnterOptionsMenu((rom char*)"Relay System?", (byte*)&relayEnable, (char*)&relayString[0], (char*)&updateRelayString, 2);
}

void setDetectionPoints(int i)
{	
	double f;
	f=detectionPoints;
	doEnterNumberMenu((rom char*) "Detection Pnts.?", 1.0, 16.0, 0, (double*)&f);
	detectionPoints=(double)((byte)(f+0.5));
}

void setLogError(int i)
{	
	doEnterNumberMenu((rom char*) "Log Error (%)?", 0.0, 1000.0, 2, (double*)&logError);
}

void setCh1(int i)
{
	doEnterOptionsMenu((rom char*) "Ch.1 Logs?", (byte*)&logChannel[0], (char*)&logChannelString[0][0], &updateLogChannelStrings, LOG_TOTAL+1);
	clearRLE();
}

void setCh2(int i)
{
	doEnterOptionsMenu((rom char*) "Ch.2 Logs?", &logChannel[1], &logChannelString[1][0], &updateLogChannelStrings, LOG_TOTAL+1);
	clearRLE();
}

void setCh3(int i)
{
	doEnterOptionsMenu((rom char*) "Ch.3 Logs?", &logChannel[2], &logChannelString[2][0], &updateLogChannelStrings, LOG_TOTAL+1);
	clearRLE();
}

void setCh4(int i)
{
	doEnterOptionsMenu((rom char*) "Ch.4 Logs?", &logChannel[3], &logChannelString[3][0], &updateLogChannelStrings, LOG_TOTAL+1);
	clearRLE();
}

void setLogPeriod(int i)
{
	doEnterNumberMenu((rom char*) "Smpl. Prd.(sec)?", 0.2, 86400.0, 1, &logPeriod);
	clearRLE();
}

void setChemistry(int i)
{
	doEnterOptionsMenu((rom char*) "Chemistry?", &cellType, &cellTypeString[0], &writeCellTypeString, 2); 
}

void setLogMode(int i)
{
	doEnterOptionsMenu((rom char*) "Log Mode?", &RLEMode, &RLEString[0], &writeRLEString, 2); 
	clearRLE();
}

void setMinVoltage(int i)
{
	doEnterNumberMenu((rom char*) "Min. Volt.(V)?", 0.0, 60.0, 1, &minVoltage);
}

void setMaxVoltage(int i)
{
	doEnterNumberMenu((rom char*) "Max. Volt.(V)?", 0.0, 60.0, 1, &maxVoltage);
}

void resetTimeOut(void)
{
	timeOutSecondsCounter=timeOutInSeconds;
}

void setTimeOutSeconds(int i)
{
	doEnterNumberMenu((rom char*) "Timeout (sec)?", 0.0, 86400.0, 0, &timeOutInSeconds);
	resetTimeOut();
}

void copyMenuString(byte menuIndex, rom MENU* myMenu, char* outString, byte shiftIndex, byte totalLength)
{
	byte i, j;
	char *str;
	char c;
	rom char* title;
	char n;
	
	if(myMenu->menu[menuIndex].body.function!=0)c='['; else c='<';
	*outString++=c;
	*outString++=0x31+menuIndex;
	*outString++=':';
	i=0;
	title=myMenu->menu[menuIndex].body.title;
	while((*title)!='\0')tempbuffer[i++]=*title++;
		// i now points to the end point and the string is copied...	 
		 if(myMenu->menu[menuIndex].body.setting!=0)
		 {
		 // if not a null pointer, then...
		 tempbuffer[i++]=':';
		 tempbuffer[i++]=' ';
		 n=myMenu->menu[menuIndex].body.numdec;
		 if(n<0)n=-n;
		 j=printFSignedDecimal(*(myMenu->menu[menuIndex].body.setting), n, 0);
		 if((myMenu->menu[menuIndex].body.numdec)<0)j++;
		 while(buffer[j]!='\0')tempbuffer[i++]=buffer[j++];
		 } else
		 if(myMenu->menu[menuIndex].body.string!=0)
		 {
		 // a string type
		 tempbuffer[i++]=':';
		 tempbuffer[i++]=' ';
		 str=myMenu->menu[menuIndex].body.string;
		 while((*str)!='\0')tempbuffer[i++]=*str++;
		 }
	while(i<(MENU_STRING_LENGTH_XTENDED))tempbuffer[i++]=' ';
	i=3;
	while(i<(totalLength-1))
			{
			j=i+shiftIndex-3;
			while(j>=(MENU_STRING_LENGTH_XTENDED))j-=(MENU_STRING_LENGTH+12);
			*outString++=tempbuffer[j];
			i++;
			}
	if(myMenu->menu[menuIndex].body.function!=0)c=']'; else c='>';
	*outString++=c;
}

void printMenuAt(byte cursor, byte menuIndex, rom MENU* myMenu, byte shiftIndex, byte totalLength)
{
	copyMenuString(menuIndex, myMenu, (char*)&lcdScreen[cursor], shiftIndex, totalLength);
}

void printMenu(byte menuIndex, rom MENU* myMenu, byte shiftIndex)
{
	printMenuAt(0, menuIndex, myMenu, shiftIndex, 16);
	printMenuAt(16, menuIndex+1, myMenu, shiftIndex, 16);
	updateLCD();
}

//*****************************************************************************
// Menu Stack
//*****************************************************************************
rom MENU* popMenu(byte* startIndex)
{
	rom MENU* g;
	if(menuFull>0)
    	{
	    menuPtr--;
		menuFull--;
		g=menuStack[menuPtr];
		(*startIndex)=menuStartIndexStack[menuPtr];
	    } else 
		{
		g=0;
		(*startIndex)=0;
		}
	return g;
}

void pushMenu(rom MENU* k, byte startIndex)
{
    if(menuFull<MENU_STACK_SIZE)
     {
	 menuStack[menuPtr]=k;
	 menuStartIndexStack[menuPtr]=startIndex;
	 menuPtr++;
	 menuFull++;
	 }    
}    

void initMenuStack(void)
{
    menuFull=0;   
    menuPtr=0;
}    

rom MENU* executeMenu(byte startIndex, rom MENU* myMenu)
{
	byte shiftIndex;
	byte exit;
	byte i, j;
	byte cursor;
	byte keyPressed;
	int length;
	char buffer[MENU_STRING_LENGTH+4];
	rom MENU* returnPointer;
	
	initKeys();			// flush any keys in the queue
	shiftIndex=0;
	exit=0;	
	returnPointer=0;
	while(!exit)
	{
		printMenu(startIndex, myMenu, shiftIndex);
		shiftIndex++;
		if(shiftIndex>MENU_STRING_LENGTH)shiftIndex=0;
		if(keyFull>0)
		{
			keyPressed=getKey();
			switch(keyPressed)
			{
				case '0':
				    break;
				case '1':
				case '2':
				case '3':
				case '4':
				case '5':
				case '6':
				case '7':
				case '8':
				case '9':
					i=keyPressed-0x31;
					j=i-startIndex;
					if((j==0)||(j==1))exit=1;
					keyPressed=i;
					break;

				case '*':
                    break;
				
				case 'A':
					if(startIndex>0)startIndex--;
					shiftIndex=0;
					break;

				case 'D':
				case 'C':
					shiftIndex=0;
					break;
				
				case 'B':
					if(startIndex<(myMenu->numMenus-2))startIndex++;
					shiftIndex=0;
					break;
				
				case '#':
					exit=2;
					keyPressed=startIndex;
					lcdTransitionType=-1;
					updateLCD();
					break;
			}
		} else delayMs(SCROLL_DELAY/2);
	}

	if(exit==1)
	{
	
	// if a leaf, execute the command
	if(myMenu->menu[keyPressed].body.function!=0)
		{
		lcdTransitionType=1;
		updateLCD();
		clearLCD();
		myMenu->menu[keyPressed].body.function(1);		// 1= blocking!
		lcdTransitionType=-1;
		updateLCD();
		pushMenu(myMenu, startIndex);
		}
		else 
		{
		returnPointer=(rom MENU*)myMenu->menu[keyPressed].nextItem;
		}
	} 
	return returnPointer;
}

/*
void executeMenuStack(void)
{
	rom MENU* x;
	rom MENU* next;
	byte startIndex;
	//
	x=popMenu(&startIndex);
	saveSettings();
	if(x!=0)
	{
		next=executeMenu(startIndex, x);
		if(next==0)
		{
    		executeMenuStack();	
        }  		
	    else
		{
		lcdTransitionType=1;
		updateLCD();
		pushMenu(x, startIndex);
		pushMenu(next, 0);
		executeMenuStack();
		}
	}
}
*/

void executeMenuStack(void)
{
    // this is a non-recursive version
    rom MENU* x;
    rom MENU* next;
    byte    startIndex;
    
    x=popMenu(&startIndex);
    while(x!=0)
    {
        next=executeMenu(startIndex, x);
        if(next==0)
        {
          x=popMenu(&startIndex);
          saveSettings();
        } else
        {
          lcdTransitionType=1;
          updateLCD();
          pushMenu(x, startIndex);
          startIndex=0;
          x=next;   
        }
    }
}    

void writeCellTypeString(byte cType)
{
	if(cType==0)
	{
	cellType=0;
	cellTypeString[0]='N';
	cellTypeString[1]='i';
	cellTypeString[2]='c';
	cellTypeString[3]='k';
	cellTypeString[4]='e';
	cellTypeString[5]='l';
	cellTypeString[6]='\0';
	} else
	{
	cellType=1;
	cellTypeString[0]='L';
	cellTypeString[1]='e';
	cellTypeString[2]='a';
	cellTypeString[3]='d';
	cellTypeString[4]=' ';
	cellTypeString[5]='A';
	cellTypeString[6]='c';
	cellTypeString[7]='i';
	cellTypeString[8]='d';
	cellTypeString[9]='\0';
	}
}

void writeFuseString(byte fuseVal)
{
	if(fuseVal==0)
	{	
		currentFuse=0;
		fuseString[0]='O';
		fuseString[1]='p';
		fuseString[2]='e';
		fuseString[3]='n';
		fuseString[4]='\0';
	} else
	{
		currentFuse=1;
		fuseString[0]='S';
		fuseString[1]='h';
		fuseString[2]='o';
		fuseString[3]='r';
		fuseString[4]='t';
		fuseString[5]='\0';
	}
}

void writeRLEString(byte rleMode)
{
	if(rleMode==0)
	{
	RLEMode=0;
	// overwrite mode, or continuous mode
	RLEString[0]='O';
	RLEString[1]='v';
	RLEString[2]='r';
	RLEString[3]='.';
	RLEString[4]='\0';
	} else
	{
	RLEMode=1;
	// capture mode or do not overwrite full buffer mode
	RLEString[0]='C';
	RLEString[1]='p';
	RLEString[2]='t';
	RLEString[3]='.';
	RLEString[4]='\0';
	}
}

void setBrightnessLevel(double percent)
{
	if(percent<0.0)percent=0.0; else if(percent>100.0)percent=100.0;
	brightnessPercent=percent;
	setBrightness=0xFE & ((byte)(2.55*percent));
}

void initValues(void)
{
	netCurrentSign=2;
	writeFuseString(1);
	lastUSB=2;						// any value different from 0 and 1
	rX=DEFAULT_RX;					// in Ohms	typically 0.0320, represents the resistance of the wire connecting the batt+ terminal on the shunt to the PC board
	alarmOn=0;
	standBy=0;
	enableStandByDetect=0;
	lcdTransitionType=0;
	firmwareVersion=VERSION;
	logPeriodCounter=0.0;
	cycles=0.0;
	icycle=1;						
	batteryPercent=0.0;;			// battery percent charge
	//batteryCapacityAH=0.0;;			// battery capacity in AH (Amp Hours)
	batteryVoltage=0.0;				// in Volts
	loadCurrent=0.0;				// in Amps
	circuitCurrent=0.0;				// in mAmps
	netCurrent=0.0;					// net current drain in Amps
	batteryMinutes=0.0;				// remaining Capacity
	lastChargeCurrent=100.0;		// deliberately large, 100A
	lastBatteryVoltage=70.0;		// deliberately larger than max voltage of 60.0
	lastSync=-1.0;					// deliberately negative representing "N/A"
	chargeCurrent=0.0;				// charging current in A
	loadDiffAmpVoltage=0.0;
	chargeDiffAmpVoltage=0.0;
	circuitVoltage=0.0;
	currentMaxTimeOut=0.0;
	brightness=0;
	loadDifferentialGain    =(DEFAULT_DIFFERENTIAL_GAIN);
	chargeDifferentialGain  =(DEFAULT_DIFFERENTIAL_GAIN);
	senseResistance         =(DEFAULT_SENSE_RESISTANCE);
	shuntResistance         =(DEFAULT_SHUNT_RESISTANCE);        // these are all hardware specific settings not restored to defaults!
	voltageRail50=(DEFAULT_VCC);
	RLETime=0.0;
}

void initDefaults(void)
{
	relayCoilResistance=(DEFAULT_COILRESISTANCE);
	cyclePercent=(DEFAULT_CYCLEPERCENT);
	detectionPoints=(DETECTION_POINTS);
	loadCurrentMax=(DEFAULT_LOADCURRENTMAX);					// overcurrent protection for relay off...
	currentMaxTimeOutPeriod=(DEFAULT_CURRENTMAXTIMEOUTPERIOD);
	relayHysteresis=(DEFAULT_RELAYHYSTERESIS);					// voltage hysteresis for relay on off automatic control
	relayOffVoltage=(DEFAULT_RELAYOFFVOLTAGE);					// voltage below which relay turns off
	relayOn=(DEFAULT_RELAYON);									// 0= on/off control
	updateRelayString(DEFAULT_RELAYENABLE);						// global enable disable for relay.
	avgSampleCount=(DEFAULT_AVG_SAMPLE_COUNT);
	updateBeeperString(1);
	logChannel[0]=0;
	logChannel[1]=0;
	logChannel[2]=0;
	logChannel[3]=0;
	cutOffVoltage=0.0;
	alarmCapacity=0.0;
//	voltageRail33=DEFAULT_VUSB;
	dividerLow=(DEFAULT_LOW_DIVIDER);
	dividerHigh=(DEFAULT_HIGH_DIVIDER);
	chargingEfficiency=(DEFAULT_CHARGING_EFFICIENCY);
	displayMode=(MODE_DEFAULT);
	logError=(DEFAULT_LOG_ERROR);
	logFloor=(DEFAULT_LOG_FLOOR);
	logPeriod=(DEFAULT_LOG_PERIOD);
	brightnessPercent=(DEFAULT_BRIGHTNESS_PERCENT);
	timeOutInSeconds=(DEFAULT_TIMEOUT_IN_SECONDS);
	fullCapacity=(DEFAULT_FULLCAPACITY);
	peukertsConstant=(DEFAULT_PEUKERTS);
	minVoltage=(DEFAULT_MINVOLTAGE);
	maxVoltage=(DEFAULT_MAXVOLTAGE);
	writeCellTypeString(DEFAULT_CHEMISTRY);
	writeRLEString(0);
	currentThreshold=(DEFAULT_THRESHOLD);
	trickleCurrent=(DEFAULT_TRICKLE_CURRENT);
	checkPeriod=(CHECK_PERIOD);
}

void updateSettings(void)
{
	resetTimeOut();
	setBrightnessLevel(brightnessPercent);
	updateBeeperString(beepOn);
	updateRelayString(relayEnable);
	writeCellTypeString(cellType);
	writeRLEString(RLEMode);
	writeFuseString(currentFuse);
	updateLogChannelStrings(0);
}

byte waitForYesOrNoResponse(void)
{
	byte key;	
	writeStringLCD(16, (rom char*)"*: Yes   #: No  ");
	updateLCD();
	key=0;
	while((key!='*')&&(key!='#'))
	{
		flushAndWaitForKey();
		key=getKey();
	}
	if(key=='*')
	{
		writeStringLCD(24, (rom char*)"        ");
		updateLCD();
		doBeep(1000);
		return 1;
	} else
	{
		writeStringLCD(16, (rom char*)"        ");	
		updateLCD();
		//delayMs(500);
	}
	return 0;
}

byte areYouSure(void)
{
	clearLCD();
	writeStringLCD(0, (rom char*)"Proceed?");
	return waitForYesOrNoResponse();
}

void restoreAllDefaults(void)
{
	if(areYouSure())
	{	
	initDefaults();
	saveSettings();
	WriteEEPROM(0, 0x76);
	}
}

void turnPowerOff(int i)
{
	if(USBSENSE)
	{
		clearWithoutUpdate();
		writeStringLCD(0, (rom char*)"USB is Connected");
		writeStringLCD(16, (rom char*)"Cannot Shut Down");
		updateLCD();
		doBeep(200);
		delayMs(2000);
	} else
	{
		if(areYouSure())
		{
			saveSettings();
			powerOff();
		}
	}
}

void declareFullCapacity(void)
{
	if(areYouSure())
	{
		while(totalUpdated==0);
		capacityAs=fullCapacity*3600.0;
		lastSync=-1.0;					// make N/A
	}
}

void initSequence(void)
{
	initPowerSystem();
	initRelaySystem();
	initValues();
	initSampling();
	initDefaults();
	if((ReadEEPROM(0))!=0x76)
	{
        // only set the capacity on the first POR!
		initRLE();
		saveSettings();
		WriteEEPROM(0, 0x76);
	} 
	else
	{
		restoreSettings();
	}
	updateSettings();
	//
	initBeep();
	initPWM();
	initLCD();
	initDelayTimer();
	initADC();
	initKeys();
	initMenuStack();
	initKeypad();
	IPR2bits.USBIP=1;	   // select interrupt priority
	UIR=0;
	PIR2bits.USBIF=0;
	PIE2bits.USBIE=1;		// enable USB interrupts!	(added MG)
	RCONbits.IPEN=1;		// enable prioritized interrupts
	INTCONbits.GIEL=1;		// enable low interrupts
	INTCONbits.GIE=1;		// enable all interrupts
	//USBConnect();
}

#pragma interruptlow myISR
void myISR(void)
{
	volatile byte x;

	if(PIR2bits.TMR3IF & PIE2bits.TMR3IE)
	{
		if(!alarmOn)BEEP=0;
		scanCode++;
		if(scanCode>=4)scanCode=0;
		if(brightness<setBrightness)putPWM(brightness+2); else if(brightness>setBrightness)putPWM(brightness-2);
		switch(scanCode)
		{
			case 0:
			default:
				scanLn=~0x10;
				break;
			case 1:
				scanLn=~0x20;
				break;
			case 2:
				scanLn=~0x40;
				break;
			case 3:
				scanLn=~0x80;
				break;
	}
	PORTB=(scanLn);
	_asm
		nop
		nop
		nop
		nop
	_endasm
	x=PORTB;	
	x=(~x & 0x0F);
	if((x!=0)&&(noKeys))
	{
		scanCounter=KEY_HOLD_OFF;
		BEEP=1;
		noKeys=0;
		switch(x)
		{
			case 0x01:
				scanRaw=scanCode;
				break;
			case 0x02:
				scanRaw=scanCode+4;
				break;
			case 0x04:
				scanRaw=scanCode+8;
				break;
			case 0x08:
				scanRaw=scanCode+12;
				break;
		}
		putKey(scanRaw);
	} else
	if(x==0)
	{
	if(scanCounter>0)scanCounter--; else noKeys=1;
	}
	PIR2bits.TMR3IF=0;
	}
	
	if(INTCONbits.TMR0IF & INTCONbits.TMR0IE)
	{
		if(timerDelay!=0)timerDelay--;
		INTCONbits.TMR0IF=0;
	}
}

void updateValueString(byte index, ram char *outstr)
{
	switch(index)
	{	
		case LOG_BATTV:
				*outstr++='B';
				*outstr++='a';
				*outstr++='t';
				*outstr++='t';
				*outstr++='.';
				*outstr++='V';
				break;
		case LOG_TIME:
				*outstr++='T';
				*outstr++='i';
				*outstr++='m';
				*outstr++='e';
				break;
		case LOG_LOADA:
				*outstr++='L';
				*outstr++='o';
				*outstr++='a';
				*outstr++='d';
				*outstr++='.';
				*outstr++='A';
				break;
		case LOG_CHARGEA:
				*outstr++='C';
				*outstr++='h';
				*outstr++='r';
				*outstr++='g';
				*outstr++='.';
				*outstr++='A';
				break;
		case LOG_CIRCUITA:
				*outstr++='C';
				*outstr++='i';
				*outstr++='r';
				*outstr++='c';
				*outstr++='.';
				*outstr++='A';
				break;
		case LOG_NETA:
				*outstr++='N';
				*outstr++='e';
				*outstr++='t';
				*outstr++='.';
				*outstr++='A';
				break;
		case LOG_CAPACITYAH:
				*outstr++='C';
				*outstr++='p';
				*outstr++='t';
				*outstr++='y';
				*outstr++='.';
				*outstr++='A';
				*outstr++='H';
				break;
		case LOG_CAPACITYPERCENT:
				*outstr++='C';
				*outstr++='p';
				*outstr++='t';
				*outstr++='y';
				*outstr++='.';
				*outstr++='%';
				break;
		case LOG_RELAYA:
				*outstr++='R';
				*outstr++='l';
				*outstr++='y';
				*outstr++='.';
				*outstr++='A';
				break;
		default:
				*outstr++='N';
				*outstr++='o';
				*outstr++='n';
				*outstr++='e';
				break;
	}
	*outstr='\0';
}

void updateLogChannelStrings(byte x)
{
	byte i;
	for(i=0; i<RLE_CHANNELS; i++)
	{
	updateValueString(logChannel[i], &logChannelString[i][0]);
	}
}

double getValue(byte index)
{
	double f;

	switch(index)
	{	
		case LOG_BATTV:
				f=batteryVoltage;
				break;
		case LOG_TIME:
				f=batteryMinutes;
				break;
		case LOG_LOADA:
				f=loadCurrent;
				break;
		case LOG_CHARGEA:
				f=chargeCurrent;
				break;
		case LOG_CIRCUITA:
				f=circuitCurrent;
				break;
		case LOG_NETA:
				f=netCurrent;
				break;
		case LOG_CAPACITYAH:
				f=batteryCapacityAH;
				break;
		case LOG_CAPACITYPERCENT:
				f=batteryPercent;
				break;
		case LOG_RELAYA:
				f=relayCurrent;
				break;
		default:
				f=0.0;
				break;
	}
	return f;
}

#pragma interrupt myISRHigh
void myISRHigh(void)
{
	volatile int x;
	volatile byte i;
	volatile double f, g, h;
	volatile byte j;
	//

	if(PIR1bits.TMR1IF & PIE1bits.TMR1IE)
	{
	if(samplingIndex<(NUM_SAMPLES-1))samplingIndex++; else samplingIndex=0;
	if(samplingCount<(SAMPLING_COUNT-1))samplingCount++; else samplingCount=0; 
	i=samplingIndex;
	// now update samples...
	switch(i)
	{
			case ANCH_VBATTNEG:
			//case ANCH_VUSB:
			case ANCH_VBATTPOS:
			case ANCH_INEG:
			case ANCH_IPOS:
					sample[i].updated=0;									// clear updated flag, ie, data in process of being changed!
					sample[i].lastSampled=samplingCount;					// keep track of sequence number
					if(i==ANCH_VBATTPOS)f=dividerHigh;
					else if(i==ANCH_VBATTNEG)f=dividerLow;
					else f=1.0;
					g=f*(((double)(readAN(i)))*voltageRail50/1023.0);		// convert ADC value to a voltage in Volts
					sample[i].value+=g;
					sample[i].value/=2.0;									// value is the instantenous value, running average...
					// Start of Averaging Section
					sample[i].totalValue+=g;
					sample[i].avgCount++;
					if(sample[i].avgCount>=((byte)avgSampleCount))
					{
						sample[i].avgValue=(sample[i].totalValue/((double)sample[i].avgCount));
						sample[i].avgCount=0;
						sample[i].totalValue=0.0;
					}
					sample[i].updated=1;				// done!
					break;

			case COMPUTE_STATE:
					// this is the last state before the cycle repeats 
					// In this state, we compute all totals...
					totalUpdated=0;

					// Primary Values
					if(((byte)avgSampleCount)==0)
					{
					batteryVoltage			=sample[ANCH_VBATTPOS].value;
					circuitVoltage			=sample[ANCH_VBATTNEG].value;
					loadDiffAmpVoltage		=sample[ANCH_INEG].value;
					chargeDiffAmpVoltage	=sample[ANCH_IPOS].value;
					//vusbRail				=sample[ANCH_VUSB].value;
					} else
					{
					batteryVoltage			=sample[ANCH_VBATTPOS].avgValue;
					circuitVoltage			=sample[ANCH_VBATTNEG].avgValue;
					loadDiffAmpVoltage		=sample[ANCH_INEG].avgValue;
					chargeDiffAmpVoltage	=sample[ANCH_IPOS].avgValue;
					//vusbRail				=sample[ANCH_VUSB].avgValue;
					}
					// Secondary Values
					// Start to compute circuit current in mA
					f=1000.0*((batteryVoltage-circuitVoltage)/senseResistance);
					circuitCurrent=f;
					// Compute Correction Factor
					h=(circuitCurrent/1000.0)*rX;
					// Start to compute load current
					g=loadDiffAmpVoltage/loadDifferentialGain;
					if(g>=0.001)g+=h;						// g is corrected voltage at amp input
					f=((g*1000.0)/(shuntResistance));
					loadCurrent=f;
					// Start to compute charge current
					g=chargeDiffAmpVoltage/chargeDifferentialGain;
					//if(g>=0.001)g-=h;
					f=((g*1000.0)/(shuntResistance));
					chargeCurrent=f;
					// Start to compute net current
					if(RELAYON==0)f=((batteryVoltage/relayCoilResistance)*1000.0); else f=0.0;
					if(f>=1.0)relayCurrent=f; else relayCurrent=0.0;
					f=(chargeCurrent-loadCurrent-((circuitCurrent+relayCurrent)/1000.0));
					netCurrent=f;
					// Compute the effect on the Battery Capacity and Time Remaining...
					if(netCurrent>=0.0005)
					{
						capacityAs+=(netCurrent*(chargingEfficiency/100.0)*(DELTAT));
						if(netCurrentSign)batteryMinutes+=(DELTAT/60.0); else batteryMinutes=0.0;
						netCurrentSign=1;
					}
					else
					if(netCurrent<=-0.0005)
					{
    					if(cellType==1)f=exp(peukertsConstant*(log(-netCurrent))); else f=-netCurrent;  // Peukert's applies to Lead Acid only!
						capacityAs+=(-f*DELTAT);
						batteryMinutes=capacityAs/(60.0*f);
						netCurrentSign=0;
					}
					// Start to compute some dependent values...
					batteryCapacityAH=capacityAs/3600.0;
					wattHours=(batteryCapacityAH*batteryVoltage);
					loadWatts=(loadCurrent*batteryVoltage);
					// calculate battery percent and cycles
					batteryPercent=100.0*(batteryCapacityAH/fullCapacity);
					f=100.0-cyclePercent;
					if(batteryPercent>=f)j=0;
					else
					if(batteryPercent<=cyclePercent)j=2;
					else j=1;
					if(j!=icycle)
					{
						if((j==0)||(j==2))
						{
							cycles+=0.5;
							WriteEEPROMF(CYCLES_ADR, (byte*)&cycles);
						}
						icycle=j;
					}
					// Check for low capacity alarm
					if(batteryPercent<alarmCapacity)
					{
						if(samplingCount & ALARM_PERIOD)
						{
							if(BEEP==0)alarmOn|=1;
						} else
						{
							alarmOn&=~1;
						}
					} else alarmOn&=~1;
					if((alarmOn!=0)&&(beepOn==1))BEEP=1; else BEEP=0;
					// Start Detection of full or minimum charge depending on chemistry...
					if(lastSync>=0.0)lastSync+=(DELTAT/60.0);
					if(cellType==0)
					{
						// this is for Nickel Chemistry
						f=netCurrent;
						if(((batteryVoltage<=lastBatteryVoltage)&&(lastBatteryVoltage<=minVoltage))||((f<=trickleCurrent)&&(f>=currentThreshold)&&(batteryVoltage>maxVoltage)))
						{
							 time+=(DELTAT);
						} else
						{
							 time=0.0;
						}
						itime+=(DELTAT);
						if(itime>=(checkPeriod/detectionPoints))
						{
							itime=0.0;
							lastBatteryVoltage=batteryVoltage;
						}
					} else
					{
						// this is for Lead Acid
						f=netCurrent;
						if((f>=currentThreshold)&&(f<=trickleCurrent)&&(batteryVoltage>=maxVoltage))
						{
							// as it stands, the lastchargecurrent is not strictly needed, because we are not detecting monotonic decreasing current
							// as in the case of Nickel batteries above...
							time+=(DELTAT);
						} else
						{	
							time=0.0;
						}
						itime+=(DELTAT);
						if(itime>=(checkPeriod/detectionPoints))
						{
							itime=0.0;
							lastChargeCurrent=f;
						}
					}
					
					if(time>=checkPeriod)
					{
						time=0.0;
						if(cellType==0)
						{
							if(batteryVoltage<=((minVoltage+maxVoltage)/2.0))
								{
								  if(icycle!=0)
								  {
								  capacityAs=0.0; 
								  lastSync=0.0;
								  }	
								} 
								else
								{
								  if(icycle!=2)
								  {
								  capacityAs=fullCapacity*3600.0;
								  lastSync=0.0;
								  }
   								 }
						} else
						if((cellType!=0)&&(icycle!=2))
						{
							capacityAs=fullCapacity*3600.0;
							lastSync=0.0;
						}
					}
					//
					checkStandBy();
					if((standBy==0)||(BUTTON==0))
					{
						if(BUTTON==0)
						{
							putPWM(2.55*brightnessPercent);
							if(stby!=0)
							{
								stby=0;
								updateLCD();
							}
						} else
						{
							if(timeOutSecondsCounter>0.0)
							{
								timeOutSecondsCounter-=(DELTAT);
								if(timeOutSecondsCounter<=0.0)
								{
								setBrightness=0;
								} 
							}
						}
						stby=0;
					} else
					{
						putPWM(0);
						if(((samplingCount & STANDBY_PERIOD)==0))
						{
						writeStringLCDDirect(0x80, (rom char*)"    Standby     ");
						writeStringLCDDirect(0xC0, (rom char*)"                ");
						}
					}
					totalUpdated=1;
					break;

			case RLE_STATE:
					(void)usbSense();
					if(logPeriodCounter>0.0)logPeriodCounter-=DELTAT;
					else
					{	
						logPeriodCounter=logPeriod;
						for(i=0; i<RLE_CHANNELS; i++)
						{
							j=logChannel[i];
							if(j!=0)
							{
								pushRLEData(i, getValue(j), logError, logFloor);
							}					
						}
					}
					if((RLEMode==1)&&(RLEFull>=RLE_BUFFER_SIZE))
					{
    				    // buffer is full & in capture mode, so add to last time...
    				    RLETime+=(DELTAT);	
    	            } else RLETime=0.0;
					// now we check the relay system
					//
					// if the relay is on, and either:
					//			(a) the voltage is less than relayOffVoltage then we turn it off; or if
					//			(b) the load current exceeds the loadCurrentMax for the currentMaxTimeOutPeriod of time, then we turn it off;
					// else:
					// if the relay is off & the voltage is above the sum of relayOffVoltage+relayHysteresis then we turn the relay on.
					//
					if((BUTTON==0)&&(currentFuse==0)){	writeFuseString(1); currentMaxTimeOut=0.0; }
					if(loadCurrent>=loadCurrentMax)currentMaxTimeOut+=(DELTAT); else currentMaxTimeOut=0.0;
					j=0;
					if(batteryVoltage>=(relayOffVoltage+relayHysteresis))
					{
						if(currentFuse==1)j=1;	// turn on
					}
					if(currentMaxTimeOut>currentMaxTimeOutPeriod)
					{
						currentMaxTimeOut=currentMaxTimeOutPeriod;
						j=2; 	// turn off due to current overload
						writeFuseString(0);
					}
					if(batteryVoltage<=relayOffVoltage)
					{
						j=2;	// turn off
					}
					
					if((j==1)&&(relayOn==0))
					{
						relayOn=1;
						currentMaxTimeOut=0.0;
					} else
					if((j==2)&&(relayOn==1))
					{
						relayOn=0;
					}
					updateRelayState();
					break;
	}
	PIR1bits.TMR1IF=0;
	}

	if(UIR & UIE)
	{
	// this service routine must be in the same priority as TMR1 and after it. This is to avoid
	// incoherencies in the RLE buffer that TMR1 ISR writes and the USB ISR reads...
	USBDriverService();
	BootService();
	PIR2bits.USBIF=0;
	}
}

/*
int waitUpdated(byte index)
{
		// blocks until the sample i has been updated and returns with maximum time before next update
		while(samplingIndex!=index);
		while(sample[index].updated==0);
		return 1;
}
*/

void ReadEEPROMF(byte address, byte* ptr)
{
	// get a floating point from EEPROM memory at address address
	// 4 bytes long (float)
	*ptr=ReadEEPROM(address);
	*(ptr+1)=ReadEEPROM(address+1);
	*(ptr+2)=ReadEEPROM(address+2);
	*(ptr+3)=ReadEEPROM(address+3);
}

void WriteEEPROMF(byte address, byte* ptr)
{
	// store a floating point double to EEPROM memory at address address
	// 4 bytes long (float)
	WriteEEPROM(address, 	(*ptr));
	WriteEEPROM(address+1, 	(*(ptr+1)));
	WriteEEPROM(address+2, 	(*(ptr+2)));
	WriteEEPROM(address+3, 	(*(ptr+3)));
}


double* settingsList[NUM_SETTINGS]=
{
	&cycles,						// this must be first...
	&loadDifferentialGain,
	&chargeDifferentialGain,
	&shuntResistance,
	&senseResistance,
	&timeOutInSeconds,
	&chargingEfficiency,
	&fullCapacity,
	&voltageRail50,
	&dividerLow,
	&dividerHigh,
	&peukertsConstant,
	&minVoltage,
	&maxVoltage,
	&brightnessPercent,
	&cutOffVoltage,
	&alarmCapacity,
	&logPeriod,
	&logError,
	&logFloor,
	&currentThreshold,
	&trickleCurrent,
	&rX,
	&relayCoilResistance,
	&checkPeriod,
	&avgSampleCount,
	&relayOffVoltage,
	&relayHysteresis,
	&loadCurrentMax,
	&currentMaxTimeOutPeriod,
	&detectionPoints,
	&cyclePercent
};

byte* byteSettingsList[NUM_BYTESETTINGS]=
{
	&displayMode,						// must be first!
	&beepOn,
	&cellType,
	&RLEMode,
	&relayEnable,
	&logChannel[0],
	&logChannel[1],
	&logChannel[2],
	&logChannel[3]
};

void saveSettings(void)
{
	byte i;
	byte offset;

	offset=SETTINGS_OFFSET;
	for(i=0; i<NUM_SETTINGS; i++)
	{
		WriteEEPROMF(offset, (byte*)settingsList[i]);
		offset+=sizeof(double);
	}

	offset=BYTESETTINGS_OFFSET;
	for(i=0; i<NUM_BYTESETTINGS; i++)
	{
		WriteEEPROM(offset, (*byteSettingsList[i]));
		offset+=sizeof(byte);
	}
}

void restoreSettings(void)
{
	byte i;
	byte offset;
	byte ibuffer[4];
	
	offset=SETTINGS_OFFSET;
	for(i=0; i<NUM_SETTINGS; i++)
	{
		ReadEEPROMF(offset, settingsList[i]);
		offset+=sizeof(double);
	}

	offset=BYTESETTINGS_OFFSET;
	for(i=0; i<NUM_BYTESETTINGS; i++)
	{
		*(byteSettingsList[i])=ReadEEPROM(offset);
		offset+=sizeof(byte);
	}
}

byte getNextMode(byte thismode)
{
	byte nmode;
	if((thismode>='1')&&(thismode<='8'))nmode=thismode+1;
	else
	if((thismode>='A')&&(thismode<='C'))nmode=thismode+1;
	else
	if(thismode=='9')nmode='A';
	else
	if(thismode=='*')nmode='1';
	else
	if(thismode=='D')nmode='*';
	else nmode='1';
	return nmode;
}

void displayDisplayMode(byte mode)
{
	
	if(mode!=SINGLE_MODE_ALL)lastmode=mode;
	switch(mode)
	{
		case MODE_CURRENT:
			seeChargeCurrentAndLoadCurrent(0);
			break;
		case MODE_TIME_REMAINING:
			seeBatteryChargeAndMinutes(0);
			break;
		case MODE_VOLTAGE_CURRENT:
			seeBatteryVoltageCurrent(0);
			break;
		case MODE_CAPACITY:
			seeCapacityDouble(0);
			break;
		case MODE_ALL_SCROLLING:
			seeWatts(0);
			break;
		case MODE_CIRCUIT:
			seeCircuit(0);
			break;
		case SINGLE_MODE_CHARGE:
			seeBatteryPercent(0);
			break;
		case SINGLE_MODE_CAPACITYAH:
			seeBatteryCapacityAH(0);
			break;
		case SINGLE_MODE_VOLTAGE:
			seeBatteryVoltage(0);
			break;
		case SINGLE_MODE_LOADCURRENT:
			seeLoadCurrent(0);
			break;
		case SINGLE_MODE_CHARGECURRENT:
			seeChargeCurrent(0);
			break;
		case SINGLE_MODE_CIRCUITCURRENT:
			//seeCircuitCurrent(0);
			//seeChargeAndVoltage(0);
			seeNetBatteryDrain(0);
			break;
		case SINGLE_MODE_NETBATTERYDRAIN:
			//seeNetBatteryDrain(0);
			//seeChargeAndVoltage(0);
			seeBatteryTimeMinutes(0);
			break;
		case SINGLE_MODE_TIMEREMAINING:
			//seeBatteryTimeMinutes(0);
			seeChargeAndVoltage(0);
			break;
		case SINGLE_MODE_ALL:
			lastmode=getNextMode(lastmode);
			displayDisplayMode(lastmode);
			break;
	}
}

void setTimerDelay(unsigned int x)
{
	TMR0H=0;
	TMR0L=0;
	INTCONbits.TMR0IF=0;
	timerDelay=x;
}

void main(void)
{
	 byte key;
	 byte scrollingMode;

	 TRISC=INIT_TRISC;
	 TRISB=INIT_TRISB;
	 PORTC=0;
	 PORTB=0;
  	 delayMs(100);
	 initSequence();
	 writeStringLCD(0, (rom char*)"Battery CapacityMeter ver.");
	 disFUnsignedLCD(26, firmwareVersion, 2, 0, 4);
	 delayMs(1000);
	 clearLCD();
	 while(1)
	 {
		if(keyFull==0)
			{
			displayDisplayMode(displayMode);
			setTimerDelay(1);
			while(timerDelay>0)goToLowPowerMode();
			if(displayMode==SINGLE_MODE_ALL)
				{
				setTimerDelay(2);
				while(timerDelay>0)goToLowPowerMode();
				}
			} 
			else
			{
				key=getKey();
				if(key=='#')
				{
				enableStandByDetect=0;
				goOutOfStandBy();
				lcdTransitionType=1;
				updateLCD();
				initKeys();
				pushMenu((rom MENU*)&mainMenu, 0);
				executeMenuStack();
				lcdTransitionType=-1;
				updateLCD();
				enableStandByDetect=1;
				} else
				{
					displayMode=key;
					scrollingMode=key;
					WriteEEPROM(BYTESETTINGS_OFFSET, displayMode);
				}			
			}
			enableStandByDetect=1;
			if((batteryVoltage<cutOffVoltage)&&(USBSENSE==0))
			{
				enableStandByDetect=0;
				writeStringLCD(0, (rom char*) "Voltage Too Low ");
				writeStringLCD(16, (rom char*)"Shutting Down...");
				writeScreen((byte*)&lcdScreen[0]);
				delayMs(3000);
				clearLCD();
				powerOff();
			}
	 }
}

/** EOF main.c ***************************************************************/
